import 'package:flutter/material.dart';

void main() {
  runApp(const MyApp());
}

class MyApp extends StatelessWidget {
  const MyApp({super.key});

  @override
  Widget build(BuildContext context) {
    return MaterialApp(
      title: 'Scroll Notification Test',
      home: const ScrollNotificationDemo(),
      debugShowCheckedModeBanner: false,
    );
  }
}

class ScrollNotificationDemo extends StatefulWidget {
  const ScrollNotificationDemo({super.key});

  @override
  State<ScrollNotificationDemo> createState() => _ScrollNotificationDemoState();
}

class _ScrollNotificationDemoState extends State<ScrollNotificationDemo> {
  final ScrollController _scrollController = ScrollController();
  final ScrollController _logScrollController = ScrollController();
  final List<String> _notificationLogs = ['Scroll notifications will appear here'];

  @override
  void initState() {
    super.initState();
    _scrollController.addListener(_onScrollListener);
  }

  void _onScrollListener() {
    // This is triggered when scroll happens
    _addLog('ScrollController listener: ${_scrollController.offset.toStringAsFixed(2)}');
  }

  void _addLog(String message) {
    setState(() {
      _notificationLogs.add('$message');
    });
    // Auto-scroll to bottom to show latest notification
    WidgetsBinding.instance.addPostFrameCallback((_) {
      if (_logScrollController.hasClients) {
        _logScrollController.animateTo(
          _logScrollController.position.maxScrollExtent,
          duration: const Duration(milliseconds: 200),
          curve: Curves.easeOut,
        );
      }
    });
  }

  @override
  void dispose() {
    _scrollController.dispose();
    _logScrollController.dispose();
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        title: const Text('Scroll Notification Test'),
      ),
      body: Row(
        children: [
          // Log display area with ListView for better scrolling
          Expanded(
            flex: 1,
            child: Container(
              color: Colors.grey[100],
              padding: const EdgeInsets.all(8.0),
              child: ListView.builder(
                controller: _logScrollController,
                itemCount: _notificationLogs.length,
                itemBuilder: (context, index) {
                  return Padding(
                    padding: const EdgeInsets.symmetric(vertical: 2.0),
                    child: Text(
                      _notificationLogs[index],
                      style: TextStyle(
                        fontSize: 12,
                        color: _notificationLogs[index].contains('OVERSCROLL')
                            ? Colors.red
                            : _notificationLogs[index].contains('AT TOP') ||
                                    _notificationLogs[index].contains('AT BOTTOM')
                                ? Colors.green
                                : Colors.black,
                        fontWeight: _notificationLogs[index].contains('OVERSCROLL')
                            ? FontWeight.bold
                            : FontWeight.normal,
                      ),
                    ),
                  );
                },
              ),
            ),
          ),
          // Scrollable content area with NotificationListener
          Expanded(
            flex: 3,
            child: NotificationListener<ScrollNotification>(
              onNotification: (ScrollNotification notification) {
                _addLog(
                  '${notification.runtimeType}: '
                  'pixels=${notification.metrics.pixels.toStringAsFixed(2)}, '
                  'maxExtent=${notification.metrics.maxScrollExtent.toStringAsFixed(2)}, '
                  'minExtent=${notification.metrics.minScrollExtent.toStringAsFixed(2)}',
                );

                // Check for overscroll at bottom
                if (notification is OverscrollNotification) {
                  _addLog('OVERSCROLL DETECTED! overscroll=${notification.overscroll.toStringAsFixed(2)}');
                }

                // Check if at top
                if (notification.metrics.pixels <= notification.metrics.minScrollExtent &&
                    notification.metrics.minScrollExtent > 0) {
                  _addLog('⬆️ AT TOP - Can load more previous items');
                }

                // Check if at bottom
                if (notification.metrics.pixels >= notification.metrics.maxScrollExtent &&
                    notification.metrics.maxScrollExtent > 0) {
                  _addLog('⬇️ AT BOTTOM - Can load more next items');
                }

                // Return false to allow further propagation
                return false;
              },
              child: CustomScrollView(
                controller: _scrollController,
                physics: const AlwaysScrollableScrollPhysics(
                  parent: ClampingScrollPhysics(
                    parent: RangeMaintainingScrollPhysics(),
                  ),
                ),
                slivers: [
                  SliverList(
                    delegate: SliverChildBuilderDelegate(
                      (context, index) {
                        return Container(
                          margin: const EdgeInsets.all(8.0),
                          padding: const EdgeInsets.all(16.0),
                          color: Colors.blue[100],
                          child: Text(
                            'Item ${index + 1}',
                            style: const TextStyle(fontSize: 16),
                          ),
                        );
                      },
                      childCount: 20,
                    ),
                  ),
                ],
              ),
            ),
          ),
        ],
      ),
    );
  }
}
