﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.Utils;

namespace DXLib.UI.Layout;

/*
 * Can be used as base class for any layout helper class that draws interface components based on current device
 * type/orientation. Allows layout functionality for components that are not otherwise views.
 */
public class DXLayout
{
	/* Properties */

	// Is component currently visible on screen?
	public bool IsVisible { get; set; }

	// Programmatically set bounds, can be used BEFORE layout
	protected Rect LayoutBounds { get; private set; }

	/* Methods */

	// Sets drawing region for this layout
	public void SetLayoutBounds( Rect bounds )
	{
		LayoutBounds = bounds;
	}

	// Sets drawing region for this layout
	public void SetLayoutBounds( double x, double y, double wd, double ht )
	{
		LayoutBounds = new Rect( x, y, wd, ht );
	}

	// Allows special handling on device rotation
	public virtual void Rotate()
	{}

	// Forces layout for current device type/orientation
	public void UpdateLayout()
	{
		UpdateLayout( DXDevice.GetLayoutType() );
	}

	// Gives subclasses opportunity to make type/orientation specific changes
	public virtual void UpdateLayout( LayoutType type )
	{
		switch ( type )
		{
			case LayoutType.Landscape: Landscape(); break;
			case LayoutType.Portrait: Portrait(); break;

			case LayoutType.WideLandscape: WideLandscape(); break;
			case LayoutType.WidePortrait: WidePortrait(); break;

			case LayoutType.MobileLandscape: MobileLandscape(); break;
			case LayoutType.MobilePortrait: MobilePortrait(); break;
			
			default: break;
		}
	}

	// Layout changes specific to each possible type/orientation
	protected virtual void Landscape() {}
	protected virtual void Portrait() {}

	protected virtual void WideLandscape() {}
	protected virtual void WidePortrait() {}

	protected virtual void MobileLandscape() {}
	protected virtual void MobilePortrait() {}
}

//
