﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Layout;
using DXLib.UI.Control;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace DXLib.UI.Form;

/*
 * Simple control to allow form header to have background color and padding separate from parent form. Can optionally
 * include a right-aligned lock icon for read-only forms, or a customizable button.
 */
public class DXFormHeader : DXGridLayout
{
	/* Constants */
	private static readonly double IconSize = DXDevice.IsTablet ? 36 : 26;

	/* Events */
	public Action CustomTapped { get; set; }

	/* Properties */
	public string Title { get => titleLbl.Text; set => titleLbl.Text = value; }

	// Context specific custom button
	public DXIconButton CustomBtn { get; private set; }

	/* Fields */
	private readonly DXLabel titleLbl;
	private readonly DXIconButton lockBtn;

	/* Methods */
	public DXFormHeader( DXForm form )
	{
		bool tablet = DXDevice.IsTablet;
		
		Horizontal = LayoutOptions.Fill;
		Vertical = LayoutOptions.Fill;

		IsClippedToBounds = false;
		
		// Gray header bar
		BackgroundColor = DXColors.Light2;

		RowSpacing = 0;
		ColumnSpacing = 0;

		// Underlying label
		titleLbl = new DXLabel
		{
			TextColor = DXColors.Dark1,

			Font = DXFonts.RobotoBold,
			FontSize = tablet ? 28 : 20,

			Margin = DXDevice.IsIOS ? 0 : new Thickness( 0, -5, 0, 0 ),
			Vertical = LayoutOptions.Center
		};

		// 1 row, 2 columns
		AddStarColumn();					// 0: title
		AddFixedColumn( IconSize );			// 1: lock/custom
		
		Add( titleLbl, 0, 0 );

		// Lock icon indicates read-only status
		if ( form.IsReadOnly )
		{
			lockBtn = new DXIconButton
			{
				Resource = "lock_solid",
				IconColor = DXColors.Dark4,

				Size = IconSize,

				Horizontal = LayoutOptions.End,
				Vertical = LayoutOptions.Center,

				IsSticky = true,
				ButtonTapped = OnLockTapped
			};

			Add( lockBtn, 1, 0 );
		}
	}

	// Adds context specific custom button
	public void SetCustom( string resource )
	{
		bool ios = DXDevice.IsIOS;
		
		// Remove
		if ( resource == null )
		{
			if ( CustomBtn != null )
			{
				Remove( CustomBtn );

				CustomBtn = null;
			}
		}
		// Add
		else
		{
			if ( CustomBtn == null )
			{
				double size = (IconSize + 10);

				// Create button
				CustomBtn = new DXIconButton
				{
					Resource = resource,

					ButtonColor = DXColors.Action,
					IconColor = DXColors.Light4,
					IconScale = 0.75,

					Horizontal = LayoutOptions.End,
					Vertical = LayoutOptions.Center,

					HasShadow = ios,
					IsSticky = true,

					ButtonTapped = OnCustomTapped
				};

				CustomBtn.SetSize( size, true );
				CustomBtn.Init();
				
				Add( CustomBtn, 1, 0 );
			}
		}
	}

	// Dynamically changes custom button color
	public void SetCustomColor( Color color )
	{
		if ( CustomBtn != null )
		{
			CustomBtn.Color = color;
		}
	}

	/* Event Callbacks */

	// Shows help tooltip when lock tapped
	private void OnLockTapped()
	{
		DXTooltip.Show( lockBtn, "form.lock" );
	}

	// Callback parent form when custom button tapped
	private void OnCustomTapped()
	{
		CustomTapped?.Invoke();
	}

	/* Layout */

	// Landscape (4:3)
	protected override void Landscape()
	{
		Margin = 0;
		Padding = new Thickness( 24, 10 );
	}

	// Portrait (3:4)
	protected override void Portrait()
	{
		Margin = DXDevice.IsAndroid ? DXUtils.Top( -15 ) : 0;
		Padding = new Thickness( 24, 5 );
	}

	// Mobile landscape
	protected override void MobileLandscape()
	{
		Padding = new Thickness( 24, 0 );
	}

	// Mobile portrait
	protected override void MobilePortrait()
	{
		Padding = new Thickness( 24, 5 );
	}
}

//
