﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Layout;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace DXLib.UI.Form;

/*
 * Encapsulates the Delete, Cancel, Save buttons present on all data forms. Allows buttons to be grouped with their own
 * background color independent of parent layout. Delete is only present in Edit mode. Only Cancel is present in Read-Only
 * mode. An optional context specific icon button can also be added. 
 */
public class DXFormButtons : DXGridLayout
{
	/* Events */
	public Action DeleteTapped { get; set; }
	public Action CancelTapped { get; set; }
	public Action SaveTapped { get; set; }

	public Action CustomTapped { get; set; }

	/* Properties */
	public bool SaveEnabled { set => saveBtn.IsDisabled = !value; }

	public bool DeleteEnabled { set => deleteBtn.IsDisabled = !value; }
	public bool DeleteVisible { set => deleteBtn.IsVisible = value; }

	// Context specific custom button resource
	public string Custom { set => SetCustom( value ); }

	// Left-aligned custom button (exposed for menu anchoring)
	public DXIconButton CustomBtn { get; private set; }

	/* Fields */
	private readonly DXButton deleteBtn;
	private readonly DXButton cancelBtn;
	private readonly DXButton saveBtn;

	/* Methods */
	public DXFormButtons( bool readOnly, bool hasDelete )
	{
		// REQUIRED
		IgnoreSafeArea = true;
		
		BackgroundColor = DXColors.Light2;
		
		// Spacing
		Padding = new Thickness( 8, 5, 13, 3 );
		Margin = 0;
		
		ColumnSpacing = 10;

		AddStarColumn();

		const double btnWd = 80;

		// Delete
		deleteBtn = new DXButton
		{
			IsVisible = hasDelete,
			IsSticky = true,

			Resource = "form.delete",
			Type = DXButton.ButtonType.Negative,

			Vertical = LayoutOptions.Center,

			ButtonWd = btnWd,
			ButtonTapped = OnDeleteTapped
		};

		deleteBtn.Init();
		
		AddFixedColumn( btnWd );
		Add( deleteBtn, 1, 0 );

		// Cancel
		cancelBtn = new DXButton
		{
			IsVisible = true,
			IsSticky = true,

			Resource = "form.cancel",
			Type = DXButton.ButtonType.Neutral,

			Vertical = LayoutOptions.Center,

			ButtonWd = btnWd,
			ButtonTapped = OnCancelTapped
		};

		cancelBtn.Init();
		
		AddFixedColumn( btnWd );
		Add( cancelBtn, readOnly ? 3 : 2, 0 );

		// Save
		saveBtn = new DXButton
		{
			IsVisible = !readOnly,
			IsSticky = false,

			Resource = "form.save",
			Type = DXButton.ButtonType.Positive,

			Vertical = LayoutOptions.Center,

			ButtonWd = btnWd,
			ButtonTapped = OnSaveTapped
		};

		saveBtn.Init();
		
		AddFixedColumn( btnWd );
		Add( saveBtn, 3, 0 );

		// Init
		SaveEnabled = false;
	}

	// Adds/removes optional context specific icon button
	private void SetCustom( string resource )
	{
		// Remove
		if ( resource == null )
		{
			if ( CustomBtn != null )
			{
				Remove( CustomBtn );
			}
		}
		// Add
		else
		{
			// Create button
			CustomBtn = new DXIconButton
			{
				Resource = resource,

				IconColor = DXColors.Dark4,
				Size = 26,

				Horizontal = LayoutOptions.Start,
				Vertical = LayoutOptions.Center,

				IsSticky = true,
				ButtonTapped = OnCustomTapped
			};

			Add( CustomBtn, 0, 0 );
		}
	}

	// Resets all button sticky states
	public void Reset()
	{
		deleteBtn.Reset();
		cancelBtn.Reset();
		
		CustomBtn?.Reset();
	}

	/* Event Callbacks */

	// Delete tapped, callback registered listener
	private void OnDeleteTapped( object sender )
	{
		DeleteTapped?.Invoke();
	}

	// Cancel tapped, callback registered listener
	private void OnCancelTapped( object sender )
	{
		CancelTapped?.Invoke();
	}

	// Save tapped, callback registered listener
	private void OnSaveTapped( object sender )
	{
		SaveTapped?.Invoke();
	}

	// Custom icon button tapped, callback registered listener
	private void OnCustomTapped()
	{
		CustomTapped?.Invoke();
	}
}

//
