﻿/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using DXLib.UI.Layout;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace DXLib.UI.Form.Control;

/*
 * A form field that displays a label and icon button. Typically used to link to a sub-screen containing additional data
 * entry content related to the field.
 */
public class DXLinkField : DXFormControl
{
	/* Events */
	public Action<object> LinkTapped { get; set; }

	/* Properties */
	public string Text { set => linkTxt.Text = DXString.Get( value ); }

	/* Fields */
	private readonly DXGridLayout layout;

	private readonly DXTextButton linkTxt;
	private readonly DXIconButton linkIcn;

	/* Methods */
	public DXLinkField()
	{
		// Normal title not used
		Title = null;

		HeightRequest = 50;

		// Layout
		layout = [];

		// Text
		linkTxt = new DXTextButton
		{
			TextColor = DXColors.Dark1,
			Font = DXFonts.Roboto,
			FontSize = 17,

			Vertical = LayoutOptions.Center,
			
			IsSticky = false,
			ButtonTapped = OnTextTapped
		};

		layout.AddStarColumn();
		layout.Add( linkTxt, 0, 0 );

		const double size = 22;

		// Icon
		linkIcn = new DXIconButton
		{
			Resource = "forward",
			Size = size,

			IconColor = DXColors.Action,
			IconScale = 0.70f,

			Horizontal = LayoutOptions.End,
			Vertical = LayoutOptions.Center,

			IsSticky = false,
			ButtonTapped = OnIconTapped
		};

		linkIcn.Init();
		
		layout.AddFixedColumn( size );
		layout.Add( linkIcn, 1, 0 );
	}

	// Disables extended controls
	public override void SetDisabled( bool disabled )
	{
		base.SetDisabled( disabled );

		linkTxt.IsDisabled = disabled;
		linkIcn.IsDisabled = disabled;
	}

	/* Abstracts */

	// Insert in parent layout
	public override void Init()
	{
		AddControl( layout );
	}

	// NA
	public override object GetObject()
	{
		return null;
	}

	// NA
	public override void SetPlaceholder( bool focused )
	{}

	// Always has value
	public override bool HasValue()
	{
		return true;
	}

	// Always valid
	public override bool IsValid()
	{
		return true;
	}
	
	/* Event Callbacks */

	// User tapped text button
	private void OnTextTapped( object data )
	{
		if ( !IsDisabled )
		{
			linkIcn.ShowTap();

			// Callback listener
			LinkTapped?.Invoke( this );
			
			linkIcn.Reset();
		}
	}

	// User tapped icon button, callback listener
	private void OnIconTapped()
	{
		if ( !IsDisabled )
		{
			linkTxt.ShowTap();

			// Callback listener
			LinkTapped?.Invoke( this );
			
			linkTxt.Reset();
		}
	}
}

//
