﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.Data;

namespace DXLib.UI.Form.Control;

/*
 * Specialized selector field used for selecting a numeric height from values ranging from 4'0 to 8'0". The selector key
 * is the height in inches.
 */
public class DXHeightSelectorField : DXSelectorField
{
	/* Constants */
	public const int MinHeight = 48;
	public const int MaxHeight = 96;

	/* Properties */
	public string SelectedHeight { get => SelectedItem; set => SelectedItem = value; }

	/* Methods */
	public DXHeightSelectorField()
	{
		List<DXItem> list = BuildList();

		// Pre-create item list
		selector.LoadItems( list );
	}

	// Builds list of height items from 4'0" to 8'0"
	private static List<DXItem> BuildList()
	{
		const int count = (MaxHeight - MinHeight);
		List<DXItem> list = new( count );

		// Build each item
		for ( int i = 0; i <= count; i++ )
		{
			int ht = (MinHeight + i);

			int feet = (ht / 12);
			int inches = (ht % 12);

			// Key is inches, values is ft'in"
			list.Add( new DXItem
			{
				Index = i,
				Key = ht.ToString(),
				Value = $"{feet}'{inches}\""
			});
		}

		return list;
	}

	// Returns list key given ft'in" value
	public static string KeyFromValue( string value )
	{
		if ( value == null )
		{
			return null;
		}

		// Parse into numeric
		int feetDelim = value.IndexOf( '\'' );
		string feetStr = value[ ..feetDelim ];

		int inchesDelim = value.IndexOf( '"' );
		string inchesStr = value.Substring( (feetDelim + 1), (value.Length - inchesDelim) );

		int feet = Convert.ToInt32( feetStr );
		int inches = Convert.ToInt32( inchesStr );

		int keyInt = (feet * 12) + inches;

		// Return key
		return keyInt.ToString();
	}
}

//
