﻿/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using Syncfusion.Maui.Picker;

using DXLib.UI.Form;
using DXLib.UI.Layout;
using DXLib.UI.Container;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace DXLib.UI.Control.Picker;

/*
 * Implements a form control field for selecting an elapsed time or duration value in hour:min:sec format. The selected
 * value will be displayed in the field, while the selection control is displayed in a popup. 
 */
public class DXDurationPicker : DXTextButton
{
	/* Events */
	public Action Selected { get; set; }

	/* Properties */
	public TimeSpan? Duration { get => duration; set => SetDuration( value ); }

	/* Fields */
	private readonly DXDurationPickerPopup popup;

	// Underlying data
	private TimeSpan? duration;

	/* Methods */
	public DXDurationPicker()
	{
        HeightRequest = DXFormControl.DefaultControlHt;

        // Styling
        TextColor = DXColors.Dark1;
		Font = DXFonts.Roboto;
		FontSize = 17;
		IsSticky = true;

		// Register for events
		ButtonTapped = OnTapped;

		// Create selector control
		popup = new DXDurationPickerPopup
		{
			Selected = OnDurationSelected,
			Dismissed = OnDismissed
		};
	}

	// Sets placeholder text (duration or empty)
	public void SetPlaceholder()
	{
		SetDuration( duration );
	}

	// Updates duration display
	private void SetDuration( TimeSpan? timeSpan )
	{
		duration = timeSpan;

		Text = (duration.HasValue ? DXUtils.FromDurationShort( (TimeSpan)duration ) : null)!;
	}

	/* Event Callbacks */

	// User tapped on duration label, show popup
	private void OnTapped( object data )
	{
		popup.Duration = duration ?? TimeSpan.Zero;

		popup.Show( this );
	}

	// User selected duration, update display
	private void OnDurationSelected( TimeSpan? timeSpan )
	{
		SetDuration( timeSpan );

		Reset();

		// Callback listener
		Selected?.Invoke();
	}
	
	// User dismissed picker popup
	private void OnDismissed()
	{
		Reset();
	}
}

/*
 * Used internally to display a time picker control for selecting elapsed
 * time or duration values.
 */
internal class DXDurationPickerPopup
{
	/* Events */
	public Action<TimeSpan?> Selected { get; set; }
	public Action Dismissed { get; set; }

	/* Properties */
	public TimeSpan? Duration { get => picker.SelectedTime; set => picker.SelectedTime = value; }

	/* Fields */
	private readonly DXPopup popup;
	private readonly SfTimePicker picker;

	/* Methods */
	public DXDurationPickerPopup()
	{
		// Layout
		DXVerticalLayout layout = new()
		{
			BackgroundColor = DXColors.Light2,

			Padding = new Thickness( 0, 0, 0, 10 ),
			Spacing = 10,
			
			Horizontal = LayoutOptions.Fill,
			Vertical = LayoutOptions.Fill
		};

		// Underlying control
		picker = new SfTimePicker
		{
			Mode = PickerMode.Default,
			TextDisplayMode = PickerTextDisplayMode.Fade,
			Format = PickerTimeFormat.HH_mm_ss,

			HourInterval = 1,
			MinuteInterval = 1,
			SecondInterval = 1,
		
			Background = DXColors.Light4,
			
			// Customize column headers
			ColumnHeaderView = new TimePickerColumnHeaderView
			{
				Background = DXColors.Light2,
				DividerColor = DXColors.Dark3,
				
				TextStyle = new PickerTextStyle
				{
					TextColor = DXColors.Dark1,
					FontFamily = DXFonts.Roboto,
					FontSize = 17
				}
			},
			
			// Customize selection
			SelectionView = new PickerSelectionView
			{
				Background = DXColors.Accent1,				
				CornerRadius = 5,
				Stroke = DXColors.Dark2
			},
			
			SelectedTextStyle = new PickerTextStyle
			{
				TextColor = DXColors.Dark1,
				FontFamily = DXFonts.RobotoBold,
				FontSize = 20
			}
		};

		// OK
		DXButton okBtn = new()
		{
			Resource = "alert.ok",
			Type = DXButton.ButtonType.Action,

			ButtonTapped = OnOkTapped
		};

		okBtn.Init();
		
		layout.Add( picker );
		layout.Add( okBtn );

		// Popup
		popup = new DXPopup( layout )
		{
			IsModal = false,

			ViewWidth = 300,
			ViewHeight = 360,
			
			PopupClosed = OnPopupClosed
		};
	}

	// Shows time picker in popup dialog
	public void Show( View view )
	{
		popup.ShowFromView( view );
	}

	/* Event Callback */

	// User selected duration from popup control
	private void OnOkTapped( object sender )
	{
		popup.Hide();

		// Callback listener
		Selected?.Invoke( Duration );
	}
	
	// User dismissed popup without OK button
	private void OnPopupClosed()
	{
		Dismissed?.Invoke();
	}
}

//
