﻿/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using DXLib.UI.Layout;
using DXLib.UI.Container;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace DXLib.UI.Control;

/*
 * A simple class for displaying a small help tooltip anchored to a parent view, typically on a data entry form. 
 */ 
public static class DXTooltip
{
	/* Constants */

	// Sizing (0 = dynamic)
	private const double DefaultWd = 250;
	private const double DefaultHt = 0;

	/* Fields */
	private static View rootView;

	// All preloaded tooltip text
	private static Dictionary<string,string> tooltipTable;

	/* Methods */

	// Must be called at startup with preloaded tooltip table
	public static void Init( Dictionary<string, string> table )
	{
		tooltipTable = table;
	}

	// Returns tooltip text corresponding to specified key, null if not found
	public static string Get( string key, string insert = null )
	{
		try
		{
			string text = tooltipTable[ key ];

			// Optionally insert text
			return (insert == null) ? text : text.Replace( DXString.Replace1, insert );
		}
		catch ( KeyNotFoundException )
		{
			return null;
		}
	}

	// Shows a tooltip with specified resource text anchored to specified view
	public static void Show( View view, string resource )
	{
		Show( view, resource, DefaultWd, DefaultHt );
	}

	// Shows a resource tooltip with specified custom sizing
	public static void Show( View view, string resource, double wd, double ht )
	{
		ShowText( view, Get( resource ), wd, ht );
	}

	// Shows a tooltip with specified text anchored to specified view
	public static void ShowText( View view, string text )
	{
		ShowText( view, text, DefaultWd, DefaultHt );
	}

	// Shows a text tooltip with specified custom sizing
	public static void ShowText( View view, string text, double wd, double ht )
	{
		rootView = view;

		// Layout
		DXVerticalLayout layout = new()
		{
			BackgroundColor = DXColors.Light4,
			Padding = DXDevice.IsIOS ? 10 : new Thickness( 10, 5, 10, 20 )
		};

		// Dynamically determine size
		int newlines = DXUtils.Count( text, "\n" );
		int rows = (text.Length / 34) + newlines + 1;

		double widthRequest = (wd == 0) ? DefaultWd : wd;
		double heightRequest = (ht == 0) ? ((rows * 17) + layout.Padding.VerticalThickness) : ht;

		layout.WidthRequest = widthRequest;
		layout.HeightRequest = heightRequest;
		
		// Tooltip
		DXLabel tooltipLbl = new()
		{
			Text = text,
			TextColor = DXColors.Dark1,
			
			Font = DXFonts.Roboto,
			FontSize = 14,
			
			LineBreakMode = LineBreakMode.WordWrap,

			HorizontalOptions = LayoutOptions.Fill,
			VerticalOptions = LayoutOptions.Fill
		};

		layout.Add( tooltipLbl );

		// Display in popup anchored to view
		DXPopup popup = new( layout )
		{
			ViewWidth = widthRequest,
			ViewHeight = heightRequest,
			
			PopupClosed = OnPopupClosed
		};

		popup.ShowFromView( view );
	}

	/* Event Callbacks */

	// Most tooltips spawn from icon button, automatically reset sticky state
	private static void OnPopupClosed()
	{
		if ( rootView is DXIconButton button )
		{
			button.Reset();
		}
	}
}

//
