/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using DXLib.UI.Layout;
using DXLib.UI.Container;

namespace DXLib.UI.Control;

/*
 * Simple wrapper for MAUI Editor providing a title, border, and default styling. 
 */
public class DXEditor : DXVerticalLayout
{
	/* Constants */
	private const double FontSize = 17;
		
	/* Events */
	public Action TextChanged { get; set; }
	
	/* Properties */
	public string Text { get => editor.Text; set => editor.Text = value; }
	
	public string Title { set => titleLbl.Text = value; }
	public string Placeholder { set => editor.Placeholder = value; }

	public int MaxLength { get => editor.MaxLength; set => editor.MaxLength = value; }
	public bool HasValue => !string.IsNullOrEmpty( Text );

	// Dynamic sizing
	public double LineCount { set => border.HeightRequest = (FontSize * value); }
	
	/* Fields */
	private readonly DXLabel titleLbl;
	private readonly DXBorder border;
	private readonly Editor editor;

	/* Methods */
	public DXEditor()
	{
		// Layout
		Padding = 0;
		Margin = 0;
		Spacing = 7;
		
		Horizontal = LayoutOptions.Fill;
		Vertical = LayoutOptions.Fill;

		IgnoreSafeArea = true;

		// Title
		titleLbl = new DXLabel
		{
			TextColor = DXColors.Dark4,
			
			Font = DXFonts.Roboto,
			FontSize = 12
		};
		
		Add( titleLbl );

		// Border
		border = new DXBorder
		{
			Padding = 5,
			
			BorderWidth = 1,
			BorderColor = DXColors.Dark4,

			Elevation = 0,
			CornerRadius = 5,
			
			Horizontal = LayoutOptions.Fill,
			Vertical = LayoutOptions.Fill
		};
		
		border.Init();
		
		// Underlying control
		editor = new Editor
		{
			Keyboard = Keyboard.Create( KeyboardFlags.All ),
			IsSpellCheckEnabled = true,
			IsTextPredictionEnabled = true,

			BackgroundColor = DXColors.Light4,
			TextColor = DXColors.Dark1,
			PlaceholderColor = DXColors.Dark4,

			FontFamily = DXFonts.Roboto,
			FontAttributes = FontAttributes.None,
			FontSize = FontSize,
			
			AutoSize = EditorAutoSizeOption.Disabled,
			
			HorizontalOptions = LayoutOptions.Fill,
			VerticalOptions = LayoutOptions.Fill
		};

		editor.TextChanged += OnTextChanged;
		
		border.Content = editor;
		Add( border );
	}

	/* Event Callbacks */
	
	// User entered text, callback listener
	private void OnTextChanged( object sender, TextChangedEventArgs args )
	{
		TextChanged?.Invoke();	
	}
}

//
