﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

namespace DXLib.UI.Control.Button;

/*
 * A button that toggles between selected and unselected states. The button contains an internal text label. The
 * selected state has a fill color while the unselected state has an external border.
 */
public class DXToggleButton : DXButton
{
	/* Constants */

	// Border thickness
	private const double DefaultBorder = 2;

	// Default colors
	private static readonly Color DefaultColor = DXColors.Transparent;
	private static readonly Color DefaultSelectedColor = DXColors.Positive;

	private static readonly Color DefaultBorderColor = DXColors.Dark4;
	private static readonly Color DefaultBorderSelectedColor = DXColors.Light4;

	private static readonly Color DefaultTextColor = DXColors.Dark4;
	private static readonly Color DefaultTextSelectedColor = DXColors.Light4;

	/* Events */
	public Action<DXToggleButton> ButtonToggled { get; set; }

	/* Properties */

	// Current state
	public bool IsSelected { get => isSelected; set => SetSelected( value ); }

	// State colors
	public Color SelectedColor { get; set; }
	public Color TextSelectedColor { get; set; }

	// Border
	public Color BorderSelectedColor { get; set; }

	/* Fields */
	private bool isSelected;

	/* Methods */
	public DXToggleButton()
	{
		Padding = 0;

		// Default minimum size
		SetSize( DefaultWd, DefaultHt );

		// Underlying control
		Font = DXFonts.RobotoBold;
		IsSticky = false;

		HasShadow = false;
		Elevation = 0;

		// Defaults
		ButtonColor = DefaultColor;
		SelectedColor = DefaultSelectedColor;

		TextColor = DefaultTextColor;
		TextSelectedColor = DefaultTextSelectedColor;

		BorderWidth = DefaultBorder;
		CornerRadius = DefaultRadius;

		BorderColor = DefaultBorderColor;
		BorderSelectedColor = DefaultBorderSelectedColor;

		SetSelected( false );
		
		// Not needed
		Down -= OnDown;
	}

	// Toggles button between selected/empty states
	private void SetSelected( bool selected )
	{
		isSelected = selected;

		// Toggle colors
		Color = isSelected ? DefaultSelectedColor : DefaultColor;
		BorderColor = isSelected ? DefaultBorderSelectedColor : DefaultBorderColor;
		TextColor = isSelected ? DefaultTextSelectedColor : DefaultTextColor;
		
		Redraw();
	}
	
	// Enables/disables button interactivity (shown via opacity)
	protected override void SetDisabled( bool disabled )
	{
		IsEnabled = !disabled;
		
		Opacity = disabled ? DisabledOpacity : 1.0;
	}

	/* Event Callbacks */

	// Tap processed after full down/up
	protected override void OnTapped( object sender, MR.Gestures.TapEventArgs args )
	{
		SetSelected( !isSelected );

		// Toggle state, notify listener
		ButtonToggled?.Invoke( this );
	}
}

//
