﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.Utils;

namespace DXLib.UI.Control.Button;

/*
 * A MAUI Label used to implement tappable text with no background, border, or shadow.
 */
public class DXTextButton : MR.Gestures.Label
{
	/* Constants */
	
	// Opacities
	private const double DefaultTapOpacity = 0.30;
	private const double DisabledOpacity = 0.20;
	
	/* Events */
	public Action<DXTextButton> ButtonTapped { get; set; }

	/* Properties */
	public LayoutOptions Horizontal { set => HorizontalOptions = value; }
	public LayoutOptions Vertical { set => VerticalOptions = value; }

	// Text
	public string Resource { set => Text = DXString.Get( value ); }

	// Text color
	public Color Color { set => TextColor = value; }
	
	// Font
	public string Font { set => FontFamily = value; }
	
	// Requires manual reset after tap?
	public bool IsSticky { get; set; }
	
	// Currently enabled?
	public bool IsDisabled { get => !IsEnabled; set { IsEnabled = !value; UpdateDisabled( value ); } }

	// Associated object
	public object Data { get; set; }

	/* Inherited */

	// Text
	// TextColor
	// FontSize
	
	/* Methods */
	public DXTextButton()
	{
		// Layout options
		Padding = new Thickness( 0, 0, 0, 0 );
		
		Horizontal = LayoutOptions.Start;
		Vertical = LayoutOptions.Center;

		HorizontalTextAlignment = TextAlignment.Start;
		VerticalTextAlignment = TextAlignment.Center;
		
		// Default styling
		BackgroundColor = Colors.Transparent;
			
		// Register for events
		Down += OnDown;
		Tapped += OnTapped;
	}
	
	// Updates visual disabled state (based on opacity)
	private void UpdateDisabled( bool disabled )
	{
		IsEnabled = !disabled;
		
		Opacity = disabled ? DisabledOpacity : 1.0;
	}
	
	// Changes to tap state
	public void ShowTap()
	{
		Opacity = DefaultTapOpacity;
	}
	
	// Programatically resets sticky button
	public void Reset()
	{
		Opacity = 1.0;
	}

	/* Event Callbacks */

	// User started tap
	private void OnDown( object sender, MR.Gestures.DownUpEventArgs args )
	{
		// Show tap state
		if ( !IsDisabled )
		{
			ShowTap();
		}
	}

	// User fully completed tap
	private void OnTapped( object sender, MR.Gestures.TapEventArgs args )
	{
		if ( !IsDisabled )
		{
			// Return to normal state unless sticky
			if ( !IsSticky )
			{
				Reset();
			}

			// Callback listener after full tap
			ButtonTapped?.Invoke( this );
		}
	}
}

//
