﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Layout;

namespace DXLib.UI.Control.Badge;

/* 
 * Displays a circular badge with a numeric value centered within it. Most properties of the badge can be customized.
 */
public class DXNumberBadge : DXGridLayout
{
	/* Properties */
	public string Text { set => label.Text = value; }
	public int Number { set => label.Text = value.ToString(); }

	public Color Color { set => circle.Color = value; }
	public Color TextColor { set => label.TextColor = value; }

	public double Size { get; set; }
	public double FontSize { get; set; }

	public bool IsBold { get; set; }

	// Adjusts positioning of text within badge
	public double AdjustX { get; set; }
	public double AdjustY { get; set; }

	/* Fields */
	private readonly DXFill circle;
	private readonly DXLabel label;

	/* Bindings */

	// Text, Color
	public static readonly BindableProperty TextProperty = BindableProperty.Create( "Text", typeof( string ), typeof( DXNumberBadge ), string.Empty, BindingMode.OneWay, null, TextPropertyChanged );
	public static readonly BindableProperty ColorProperty = BindableProperty.Create( "Color", typeof( Color ), typeof( DXNumberBadge ), DXColors.Action, BindingMode.OneWay, null, ColorPropertyChanged );

	/* Methods */
	public DXNumberBadge()
	{
		Padding = 0;
		Margin = 0;

		Horizontal = LayoutOptions.Start;
		Vertical = LayoutOptions.Start;

		// Background badge
		circle = new DXFill();

		// Number
		label = new DXLabel
		{
			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center
		};

		// Defaults
		AdjustY = 0;
		IsBold = true;
	}

	// Post construction setup (size MUST be set first)
	public void Init()
	{
		Clear();

		// Circle
		circle.Circle = Size;

		// Number
		label.Margin = new Thickness( AdjustX, AdjustY, 0, 0 );
		label.Font = IsBold ? DXFonts.OswaldBold : DXFonts.Oswald;
		label.FontSize = FontSize;

		// Layout
		Add( circle, 0, 0 );
		Add( label, 0, 0);
    }

	/* Dynamic Properties */

	// Updates number on binding value change
	private static void TextPropertyChanged( BindableObject bindable, object oldValue, object newValue )
	{
		DXNumberBadge control = (DXNumberBadge)bindable;

		string value = newValue?.ToString();

		control.Text = value;
		control.IsVisible = !string.IsNullOrEmpty( value );
	}

	// Updates badge color on binding value change
	private static void ColorPropertyChanged( BindableObject bindable, object oldValue, object newValue )
	{
		DXNumberBadge control = (DXNumberBadge)bindable;

		control.Color = (Color)newValue;
	}
}

//
