﻿/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using DXLib.Log;
using DXLib.Utils;

namespace DXLib.UI.Container;

/*
 * A basic wrapper for a MAUI ScrollView.
 */
public class DXScroll : ScrollView
{
	/* Properties */
	public LayoutOptions Horizontal { set => HorizontalOptions = value; }
	public LayoutOptions Vertical { set => VerticalOptions = value; }

	/* Methods */

	// Scrolls to top of view
	public void ScrollToTop( bool animated = false )
	{
		MainThread.BeginInvokeOnMainThread( Action );
		return;

		async void Action() => await ScrollToAsync( 0, 0, animated );
	}

	// Scrolls to top of view, waiting until complete
	public async Task ScrollToTopWait( bool animated = false )
	{
		await ScrollToAsync( 0, 0, animated );
	}

	// Scrolls to bottom of view
	public void ScrollToBottom( bool animated = false )
	{
		MainThread.BeginInvokeOnMainThread( Action );
		return;

		async void Action() => await ScrollToAsync( 0, ContentSize.Height, animated );
	}

	// Vertically scrolls to specified Y position
	public void ScrollToY( double y, bool animated = false )
	{
		MainThread.BeginInvokeOnMainThread( Action );
		return;

		async void Action() => await ScrollToAsync( 0, y, animated );
	}

	// Scrolls so that specified view is completely on-screen
	public void ScrollToView( View view, bool animated = false )
	{
        if ( view != null )
        {
	        async void Action()
	        {
		        try
		        {
			        await ScrollToAsync( view, ScrollToPosition.MakeVisible, animated );
		        }
		        catch ( Exception ex )
		        {
			        DXLog.Exception( "scroll", ex );
		        }
	        }

	        MainThread.BeginInvokeOnMainThread( Action );
        }
    }

	// Force full width on mobile landscape
	public void AdjustForSafeArea( View view )
	{
		if ( DXDevice.IsIOS && DXDevice.IsMobile && DXDevice.IsLandscapeLeft() )
		{
			VerticalScrollBarVisibility = ScrollBarVisibility.Never;

			double safeLeft = DXDevice.SafeArea().Left;
			
			view.Margin = new Thickness( 0, 0, -safeLeft, 0 );
		}
	}
	
    /* Layout */

    // Allows special handling for device rotation
    public virtual void Rotate()
	{}

	// Forces layout for current device type/orientation
	public void UpdateLayout()
	{
		UpdateLayout( DXDevice.GetLayoutType() );
	}

	// Gives subclasses opportunity to make type/orientation specific changes
	public virtual void UpdateLayout( LayoutType type )
	{
		switch ( type )
		{
			case LayoutType.Landscape: Landscape(); break;
			case LayoutType.Portrait: Portrait(); break;

			case LayoutType.WideLandscape: WideLandscape(); break;
			case LayoutType.WidePortrait: WidePortrait(); break;

			case LayoutType.MobileLandscape: MobileLandscape(); break;
			case LayoutType.MobilePortrait: MobilePortrait(); break;
			
			default: break;
		}
	}

	// Layout changes specific to each possible size
	protected virtual void Landscape() {}
	protected virtual void Portrait() {}

	protected virtual void WideLandscape() {}
	protected virtual void WidePortrait() {}

	protected virtual void MobileLandscape() {}
	protected virtual void MobilePortrait() {}
}

//
