﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using Syncfusion.Maui.Popup;

using DXLib.UI.Layout;

using DXLib.UI.Form;
using DXLib.UI.Form.Control;

using DXLib.UI.Control;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace DXLib.UI.Alert;

/*
 * Displays a modal popup with a text entry field to confirm deletion. The user must type 'DELETE' for the callback
 * handler to be invoked.
 */
public static class DXDeleteAlert
{
	/* Constants */
	private const string ConfirmText = "DELETE";
	
	/* Fields */
	private static SfPopup popup;

	private static DXLabel headerLbl;
	private static DXLabel msgLbl;
	private static DXTextField textField;

	private static DXButton cancelBtn;
	private static DXButton deleteBtn;

	private static Action cancelAction;
	private static Action deleteAction;

	/* Methods */

	// Shows delete confirmation popup for specified object type
	public static void Show( string typeStr, Action delete, Action cancel = null )
	{
		string title = DXString.Get( "form.delete" );
		string msg = DXString.Get( "form.delete.msg", typeStr );

		// Object specified message
		ShowRaw( title, msg, delete, cancel );
	}
	
	// Shows delete confirmation popup
	public static void ShowRaw( string title, string msg, Action delete, Action cancel = null )
	{
		// Lazily instantiate cached popup
		if ( popup == null )
		{
			popup = new SfPopup();

			Init( title );
		}

		// Configure message
		msgLbl.Text = msg;

		// Configure textfield
		textField.Text = null;
		textField.UpdateState( false );

		// Save event callbacks
		deleteBtn.IsDisabled = true;

		deleteAction = delete;
		cancelAction = cancel;

		// Display
		popup.Show();
	}

	// Performs all one-time only setup
	private static void Init( string title )
	{
		bool ios = DXDevice.IsIOS;
		bool tablet = DXDevice.IsTablet;
		
		// Fixed size
		double wd = tablet ? 360 : 340;

		const int headerHt = 40;
		int bodyHt = ios ? 135 : 145;
		const int footerHt = 50;

		popup.HeaderHeight = headerHt;
		popup.FooterHeight = footerHt;
		popup.AutoSizeMode = PopupAutoSizeMode.Both;
		
		popup.Padding = 0;
		popup.Margin = 0;

		// Modal
		popup.StaysOpen = true;
		popup.IsFullScreen = false;

		// Header/footer
		popup.ShowHeader = true;
		popup.ShowFooter = true;
		popup.ShowCloseButton = false;

		// Quick fade animation in/out
		popup.AnimationMode = PopupAnimationMode.Fade;
		popup.AnimationEasing = PopupAnimationEasing.SinIn;
		popup.AnimationDuration = 250;		
		
		PopupStyle style = popup.PopupStyle;

		// Style
		style.CornerRadius = 4;
		style.HasShadow = true;
		style.OverlayColor = DXAlert.OverlayColor;
		style.PopupBackground = DXColors.Accent1;

		style.StrokeThickness = 1;
		style.Stroke = DXColors.Dark3;

		popup.BackgroundColor = DXColors.Accent2;
		
		// Header
		ContentView headerView = new()
		{
			Padding = new Thickness( 17, 5, 15, 0 ),
			Margin = 0,
			
			BackgroundColor = DXColors.Light4,
			
			WidthRequest = wd,
			HeightRequest = headerHt
		};

		headerLbl = new DXLabel
		{
			Text = title,

			TextColor = DXColors.Dark1,
			Font = DXFonts.RobotoBold,
			FontSize = 20,

			LineBreakMode = LineBreakMode.NoWrap,
			VAlign = TextAlignment.Center,

			Horizontal = LayoutOptions.Start,
			Vertical = LayoutOptions.Center
		};

		headerView.Content = headerLbl;

		popup.HeaderTemplate = new DataTemplate( () => headerView );

		// Body layout
		DXGridLayout bodyLayout = new()
		{
            BackgroundColor = DXColors.Light4,

			Padding = new Thickness( 17, 0, 15, 15 ),
			RowSpacing = 10,

			WidthRequest = wd,
			HeightRequest = bodyHt,
			
			Horizontal = LayoutOptions.Fill,
			Vertical = LayoutOptions.Fill
		};

		// Message
		msgLbl = new DXLabel
		{
			TextColor = DXColors.Dark1,
			Font = DXFonts.Roboto,
			FontSize = 17,

			LineBreakMode = LineBreakMode.WordWrap,
			
			Horizontal = LayoutOptions.Fill,
			Vertical = LayoutOptions.Fill
		};
		
		// Textfield
		textField = new DXTextField
		{
			IsRequired = false,
			Type = DXTextField.TextType.None,
			MinLength = 0,
			MaxLength = 32,

			Margin = DXUtils.Top( -10 ),

			TextChanged = OnTextChanged
		};

		textField.Init();

		bodyLayout.AddStarRow();
		bodyLayout.AddFixedRow( DXFormControl.DefaultControlHt );
		
		bodyLayout.Add( msgLbl, 0, 0 );
		bodyLayout.Add( textField, 0, 1 );

		popup.ContentTemplate = new DataTemplate( () => bodyLayout );

		// Footer
		DXGridLayout footerView = new()
		{
			Padding = new Thickness( 0, 10, 0, 15 ),
			Margin = 0,
			
			ColumnSpacing = 10,
			BackgroundColor = DXColors.Light4,
			
			WidthRequest = wd,
			HeightRequest = footerHt
		};

		// 3 columns
		footerView.AddStarColumn();
		footerView.AddAutoColumn();
		footerView.AddFixedColumn( 5 );

		// Delete
		deleteBtn = new DXButton
		{
			Resource = "form.delete",
			Type = DXButton.ButtonType.Negative,

			ButtonWd = 80,

			Horizontal = LayoutOptions.End,
			Vertical = LayoutOptions.Center
		};

		deleteBtn.Init();
		
		footerView.Add( deleteBtn, 0, 0 );

		// Cancel
		cancelBtn = new DXButton
		{
			Resource = "alert.cancel",
			Type = DXButton.ButtonType.Neutral,

			ButtonWd = 80,

			Horizontal = LayoutOptions.End,
			Vertical = LayoutOptions.Center
		};

		cancelBtn.Init();
		
		footerView.Add( cancelBtn, 1, 0 );

		popup.FooterTemplate = new DataTemplate( () => footerView );

		// Register for events
		deleteBtn.ButtonTapped += OnDeleteTapped;
		cancelBtn.ButtonTapped += OnCancelTapped;
	}

	/* Event Callbacks */

	// Delete tapped
	private static void OnDeleteTapped( object sender )
	{
		popup.IsOpen = false;		// Do NOT use Dismiss()

		deleteAction?.Invoke();
	}

	// Cancel tapped
	private static void OnCancelTapped( object sender)
	{
		popup.IsOpen = false;

		cancelAction?.Invoke();
	}

	// Delete button only enabled with 'DELETE' text entry
	private static void OnTextChanged()
	{
		string text = textField.Text;

		deleteBtn.IsDisabled = string.IsNullOrEmpty( text ) || !text.Equals( ConfirmText );
	}
}

//
