﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;

using DXLib.UI.Control;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Implements a horizontal strip of video player controls including buttons for play/pause, slow/fast-rewind, and
 * slow/fast-forward. All controls act on the native video player.
 */
public class VideoControls : DXGridLayout
{
	/* Constants */

	// Sizing
	private const double MinSize = 26;
	private const double MaxSize = 52;

	private const double MinFontSize = 12;
	private const double LargeFactor = 1.2;

	// Speed control
	private const float SlowSpeed = 0.5f;
	private const float NormalSpeed = 1.0f;
	private const float FastSpeed = 2.0f;

	/* Properties */
	public TimeSpan Time { set => timeLbl.Text = DXUtils.FromDurationShort( value ); }
	
	/* Fields */
	private DXIconButton fastRewindBtn;
	private DXIconButton slowRewindBtn;

	private DXIconButton playBtn;
	private DXIconButton pauseBtn;

	private DXIconButton slowForwardBtn;
	private DXIconButton fastForwardBtn;

	private DXIconButton selectedBtn;

	// '0:00'
	private readonly DXLabel timeLbl;

	// Dynamic sizing
	private double smallSize;
	private double largeSize;
	private double fontSize;

	// Parent
	private readonly VideoPlayer player;

	/* Methods */
	public VideoControls( VideoPlayer player )
	{
		this.player = player;

		Horizontal = LayoutOptions.Fill;
		Vertical = LayoutOptions.Fill;
		
		// Time
		timeLbl = new DXLabel
		{
			TextColor = DXColors.Light4,
			Font = DXFonts.OswaldBold,

			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center
		};
	}

	// Returns full pixel width of control strip
	public double GetWidth()
	{
		return ((largeSize * 2) + (smallSize * 4) + (ColumnSpacing * 5));
	}

	// Returns full pixel height of control strip
	public double GetHeight()
	{
		return (largeSize + fontSize);
	}

	// Disables specified control button
	private void Disable( DXIconButton btn )
	{
		btn.IsDisabled = true;

		// Enable previous selection
		if ( selectedBtn != null )
		{
			selectedBtn.IsDisabled = false;
		}
		
		selectedBtn = btn;
	}

	/* Event Callbacks */

	// 2x rewind
	private void OnFastRewind()
	{
		player.Speed = -FastSpeed;

		Disable( fastRewindBtn );
	}

	// Slo-mo rewind
	private void OnSlowRewind()
	{
		player.Speed = -SlowSpeed;

		Disable( slowRewindBtn );
	}

	// Play
	private void OnPlay()
	{
		player.Speed = NormalSpeed;
		player.Play();

		Disable( playBtn );
	}

	// Pause
	private void OnPause()
	{
		player.Pause();

		Disable( pauseBtn );
	}

	// Slo-mo forward
	private void OnSlowForward()
	{
		player.Speed = SlowSpeed;

		Disable( slowForwardBtn );
	}

	// 2x fast-forward
	private void OnFastForward()
	{
		player.Speed = FastSpeed;

		Disable( fastForwardBtn );
	}

	/* Layout */

	// Re-creates dynamically sized layout
	public void UpdateLayout( double wd )
	{
		ClearAll();

		Padding = 0;
		RowSpacing = 0;

		ColumnSpacing = (wd * 0.020);

		// Size proportionate to view (restrict to min/max)
		double scaled = (wd * 0.055);
		double size = Math.Max( scaled, MinSize );

		smallSize = Math.Min( size, MaxSize );
		largeSize = (smallSize * LargeFactor );

		fontSize = Math.Max( (wd * 0.022), MinFontSize );

		AddFixedRow( largeSize );
		AddFixedRow( fontSize );

		// Slow/Fast Rewind
		fastRewindBtn = CreateControl( "fastrewind", smallSize, OnFastRewind, 0, true );
		slowRewindBtn = CreateControl( "slowrewind", smallSize, OnSlowRewind, 1, true );

		// Play/Pause
		playBtn = CreateControl( "play_outline", largeSize, OnPlay, 2 );
		pauseBtn = CreateControl( "pause_outline", largeSize, OnPause, 3 );

		// Slow/Fast Forward
		slowForwardBtn = CreateControl( "play", smallSize, OnSlowForward, 4 );
		fastForwardBtn = CreateControl( "fastforward", smallSize, OnFastForward, 5 );

		// Time
		timeLbl.FontSize = fontSize;
		timeLbl.Margin = DXUtils.Top( DXDevice.IsIOS ? 0 : -(wd * 0.010) );

		Add( timeLbl, 0, 1, 6, 1 );
	}

	// Creates new video control button and adds to layout
	private DXIconButton CreateControl( string resource, double size, Action action, int column, bool rewind = false )
	{
		// Create
		DXIconButton control = new()
		{
			Resource = resource,

			IconColor = DXColors.Light4,
			Size = size,
			HasShadow = true,

			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center,

			ButtonTapped = action,

			IsDisabled = (rewind && DXDevice.IsAndroid)
		};

		control.Init();
		
		// Add to layout
		AddFixedColumn( size );
		Add( control, column, 0 );

		return control;
	}
}

//
