﻿/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using DXLib.UI.Alert;
using DXLib.UI.Control;
using DXLib.UI.Gestures;
using DXLib.UI.Container;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Base class for all tool modules (RallySync, Video). Tool display is controlled by the master UI Shell.
 */
public abstract class ToolForm : DXContent
{
	/* Constants */
	protected static readonly Color DefaultColor = VideoPlayer.DefaultColor;
	
	// Video load timeout (ms)
	protected const int LoadTimeout = (30 * 1000);

	/* Properties */

	/* Fields */
	protected readonly DXAbsoluteGestures layout;
	protected readonly DXScroll scroll;

	// Underlying data
	protected readonly Set set;
	
	/* Methods */
	protected ToolForm( Set set )
	{
		this.set = set;

		BackgroundColor = DefaultColor;

		// Master layout
		layout = new DXAbsoluteGestures
		{
			Padding = 0,
			BackgroundColor = DefaultColor,

			Horizontal = LayoutOptions.Fill,
			Vertical = LayoutOptions.Fill,

			IgnoreSafeArea = true
		};

		// Tablet fixed layout
		if ( DXDevice.IsTablet )
		{
			Content = layout;
		}
		// Mobile scrolls
		else
		{
			scroll = new DXScroll
			{
				Padding = 0,
				BackgroundColor = DefaultColor,
				Orientation = ScrollOrientation.Vertical,

				Horizontal = LayoutOptions.Fill,
				Vertical = LayoutOptions.Fill,

				Content = layout
			};

			Content = scroll;
		}
	}

	// Post construction initialization
	public virtual async Task Init()
	{
		await Task.FromResult( default( object ) );
	}

	// Preps for video playback and/or stat recording
	public virtual async Task Start()
	{
		await Task.FromResult( default( object ) );
	}

	/* Show/Hide */

	// Shows tool UI
	public void Show()
	{
		IsVisible = true;
	}

	// Hides tool UI
	public void Hide()
	{
		IsVisible = false;
	}
	
	/* Video */

	// Handles cleanup and error display following video load timeout
	protected void HandleTimeout()
	{
		DXSpinner.Stop();

		DXAlert.ShowError( "video.err", "video.err.timeout", OnToolClosed );
	}

	/* Event Callbacks */

	// User exiting tool, must be implemented by concrete class
	protected abstract void OnToolClosed();
}

//
