﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Control;
using DXLib.UI.Gestures;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Represents one row in the toolbar 'More' popup view. Each row contains an icon button and corresponding text label. 
 */
public class ToolBarItem : DXGridGestures
{
	/* Constants */
	private const double IconSize = 24;
	private const int TapDelay = 250;

	// Padding
	public const double RowPad = 4;
	private const double SidePad = 5;

	// State colors
	private static readonly Color DefaultColor = DXColors.Light4;
	private static readonly Color TapColor = DXColors.Accent1;

	/* Events */
	public Action ItemTapped { get; set; }

	/* Properties */
	public string Icon { set => icon.Resource = value; }
	public string Resource { set => label.Resource = value; }

	public Color Color { set { icon.Color = value; label.TextColor = value; } }

	/* Fields */
	private readonly DXIcon icon;
	private readonly DXLabel label;

	/* Methods */
	public ToolBarItem()
	{
		BackgroundColor = DefaultColor;

		// Padding for tap state
		Padding = new Thickness( SidePad, RowPad );
		ColumnSpacing = 10;

		Horizontal = LayoutOptions.Fill;	// REQUIRED (background fill)
		Vertical = LayoutOptions.Center;
		
		// REQUIRED
		IgnoreSafeArea = true;
		
		// 2 columns
		AddFixedColumn( IconSize );			// 0: icon
		AddStarColumn();					// 1: label

		// Icon
		icon = new DXIcon
		{
			Color = DXColors.Dark1,
			Size = IconSize
		};

		// Label
		label = new DXLabel
		{
			TextColor = DXColors.Dark1,
			Font = DXFonts.Roboto,
			FontSize = 20
		};

		Add( icon, 0, 0 );
		Add( label, 1, 0 );

		// Listen for taps
		Tapped += OnTapped;
	}

	/* Event Callbacks */

	// User tapped anywhere on item
	private void OnTapped( object sender, MR.Gestures.TapEventArgs args )
	{
		BackgroundColor = TapColor;
		
		// Show brief tap state, callback listener
		DXTimer.Delay( TapDelay, () =>
		{
			ItemTapped?.Invoke();

			BackgroundColor = DefaultColor;
		});
	}
}

//
