﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.Data;

namespace iStatVball3;

/*
 * Provides functionality for prompting the user to review the app. The prompt will only be displayed if the user has
 * recorded at least 10 matches in the season and has won the just completed match. If the user dismisses the prompt,
 * they will not be asked again for at least 30 days. Once they have chosen to review, either positive or negative, they
 * will not be prompted again.
 */
public static class ReviewEngine
{
	/* Constants */

	// Minimum days between prompts
	private const int MinDays = 30;

	// Minimum matches before prompt
	private const int MinMatches = 10;

	/* Methods */

	// Prompts user for review if applicable
	public static async Task Prompt( Match match, bool debug = false )
	{
		if ( DXData.HasConnection() )
		{
			User user = Shell.CurrentUser;
			bool? reviewed = user.Reviewed;

			// Only reviewed once
			if ( reviewed is null or false )
			{
				DateTimeOffset? requested = user.ReviewRequested;
				DateTimeOffset now = DateTimeOffset.Now;
				
				// At least one month between requests
				if ( (requested == null) || (now.Subtract( (DateTimeOffset)requested ).TotalDays > MinDays) )
				{
					// Match must have ended victoriously
					if ( debug || match.Won )
					{
						// Must have recorded at least 10 matches
						if ( debug || (match.Season.MatchCount >= MinMatches) )
						{
							ReviewPrompt prompt = [];
							prompt.Show();

							// Persist timestamp
							await user.UpdateReviewRequested();
						}
					}
				}
			}
		}
	}
}

//
