﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Container;

using DXLib.UI.Control;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace iStatVball3;

/*
 * UI control displayed at bottom of main UI shell. Represents "breadcrumb" path to currently displayed screen. Each
 * item in path is tappable to support jumping directly to any location in path. Last item in bar will always be same as
 * active screen title. Previous items will show name of object selected in path.
 */
public class PathBar : DXContent
{
	/* Constants */
	private static readonly double BaseHt = (DXDevice.IsIOS ? 15 : 30);
	private const double BasePad = 10;

	public static readonly double ViewHt = (BaseHt + (BasePad * 2));

	// Colors
	private static readonly Color ActiveColor = DXColors.Light2;
	private static readonly Color InactiveColor = DXColors.Dark4;
	private static readonly Color ArrowColor = DXColors.Dark4;

	/* Properties */
	public bool IsBarEnabled { set => Enable( value ); }

	/* Fields */
	private readonly DXScroll scroll;
	private readonly DXHorizontalLayout layout;

	private readonly List<PathBarItem> items;

	/* Methods */
	public PathBar()
	{
		Color = DXColors.Dark2;

		// Scroll area
		scroll = new DXScroll
		{
			BackgroundColor = DXColors.Dark2,
			Orientation = ScrollOrientation.Horizontal,

			Padding = 0
		};

		// Internal layout
		layout = new DXHorizontalLayout
		{
			Padding = 0,
			Spacing = 0,
			
			Horizontal = LayoutOptions.Fill,
			Vertical = LayoutOptions.Fill,

			// REQUIRED
			IgnoreSafeArea = true
		};

		scroll.Content = layout;
		Content = scroll;

		// Allocate container
		items = [];
	}

	// Removes all items from path bar
	public void ClearAll()
	{
		items.Clear();
		layout.Clear();
	}

	// Pushes new item onto bar
	public void Push( PathBarItem item )
	{
		bool ios = DXDevice.IsIOS;
		bool wide = DXDevice.IsTabletWide;

		int count = items.Count;

		// Update previous item
		if ( count > 0 )
		{
			// Add arrow
			DXIcon arrow = new()
			{
				Margin = new Thickness( 2, 0, 2, 0 ),

				Resource = "forward.png",
				Color = ArrowColor,
				Size = 15,

				Vertical = LayoutOptions.Center,
			};
			
			layout.Add( arrow );

			PathBarItem prevItem = items[ count - 1 ];
			DXTextButton prevBtn = prevItem.Button;

			// Previous text changes to inactive
			prevBtn.Text = item.InactiveText;
			prevBtn.TextColor = InactiveColor;
			
			prevBtn.IsDisabled = false;
		}

		// Add new button
		DXTextButton activeBtn = new()
		{
			IsDisabled = true,
			
			Text = item.ActiveText,
			Data = count,

			TextColor = ActiveColor,
			Font = DXFonts.RobotoBold,
			FontSize = 15,

			Padding = 5,
			Margin = ios ? 0 : DXUtils.Top( wide ? -5 : -2 )
		};

		activeBtn.ButtonTapped += OnButtonTapped;

		layout.Add( activeBtn );

		// Save new item
		item.Index = count;
		item.Button = activeBtn;

		items.Add( item );

		// Scroll to end
		scroll.ScrollToView( activeBtn );
	}

	// Pops last item off bar
	public void Pop()
	{
		if ( items.Count > 1 )
		{
			// Remove last
			InternalPop();

			int newCount = items.Count;

			PathBarItem active = items[ newCount - 1 ];
			DXTextButton activeBtn = active.Button;

			// Update new last item
			activeBtn.Text = active.ActiveText;
			activeBtn.TextColor = ActiveColor;

			activeBtn.IsDisabled = true;

			// Update previous item (if applicable)
			if ( newCount > 1 )
			{
				DXTextButton inactiveBtn = items[ newCount - 2 ].Button;

				inactiveBtn.Text = active.InactiveText;
				inactiveBtn.TextColor = InactiveColor;
			}

			// Scroll to end (mobile-portrait only)
			if ( DXDevice.IsMobilePortrait() )
			{
				scroll.ScrollToView( activeBtn );
			}
		}
	}

	// Visually enables/disables control and user input
	private void Enable( bool enabled )
	{
		IsEnabled = enabled;
		Opacity = enabled ? 1.0 : 0.3;
	}

	// Used internally to remove data for last item on bar without UI update
	private void InternalPop()
	{
		items.RemoveAt( items.Count - 1 );

		layout.RemoveLast();  	// Item	
		layout.RemoveLast();	// Arrow
	}

	/* Event Callbacks */

	// Called back when any bar item tapped
	private void OnButtonTapped( object sender )
	{
		object data = (sender as DXTextButton)?.Data;

		int index = (data == null) ? 0 : (int)data; 
		int count = items.Count;

		// Can jump back to any item in bar
		if ( index < (count - 1) )
		{
			PageStack pageStack = Shell.Instance.PageStack;

			// Remove items, pop pages, without UI refresh
			int removeCount = (count - index - 2);

			for ( int i = 0; i < removeCount; i++ )
			{
				InternalPop();
				pageStack.Pop( false );
			}

			// Update UI for last item only
			Pop();

			pageStack.Pop( false );
			pageStack.Refresh();
		}
	}
	
	/* Layout */

	// Updates layout based on orientation
	public override void UpdateLayout( LayoutType type )
	{
		base.UpdateLayout( type );

		double safeBottom = DXDevice.GetSafeBottom();

		// PathBar draws behind bottom safe area
		Padding = new Thickness( BasePad, BasePad, BasePad, (BasePad + safeBottom) );
	}
}

//
