﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;

using DXLib.UI.Control;
using DXLib.UI.Control.Image;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Displays 'Welcome' landing page in launch flow. User can branch from here to either create account or login with an
 * existing account.
 */ 
public class LaunchLanding : LaunchPage
{
	/* Fields */
	private readonly DXLabel titleLbl;
	private readonly DXImage titleImg;

	private readonly DXLabel infoLbl;
	private readonly DXIconButton helpBtn;

	private readonly DXButton createBtn;
	private readonly DXButton loginBtn;

	/* Methods */
	public LaunchLanding()
	{
		BackgroundColor = DXColors.Dark1;

		// Layout
		DXVerticalLayout layout = new()
		{
			Spacing = 15,
			Padding = new Thickness( 0, 0, 0, 30 ),

			Horizontal = LayoutOptions.Fill,
			Vertical = LayoutOptions.Center
		};

		// 'Welcome to'
		titleLbl = new DXLabel
		{
			Text = DXString.GetUpper( "launch.title" ),
			TextColor = DXColors.Light4,
			Font = DXFonts.RobotoBoldItalic,

			Margin = 0,

			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.End
		};

		layout.Add( titleLbl );

		// 'iStatVball 3'
		titleImg = new DXImage
		{
			DownsampleToViewSize = false,
			
			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center
		};

		titleImg.FromResource( "logo_launch.png" );
		
		layout.Add( titleImg );

		// 'How would you like to get started?'
		DXHorizontalLayout infoLayout = new()
		{
			Spacing = 15,

			Margin = new Thickness( 0, 25, 0, 20 ),
			Horizontal = LayoutOptions.Center
		};

		infoLbl = new DXLabel
		{
			Text = DXString.Get( "launch.info" ),
			TextColor = DXColors.Light4,
			Font = DXFonts.RobotoBold,

			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center
		};

		infoLayout.Add( infoLbl );

		// (?)
		helpBtn = new DXIconButton()
		{
			Resource = "help.png",
			
			IconColor = DXColors.Light1,
			IsSticky = true,
			Vertical = LayoutOptions.Center,
			
			ButtonTapped = OnHelpTapped
		};
		
		infoLayout.Add( helpBtn );

		layout.Add( infoLayout );

		// 'Create New Account'
		createBtn = new DXButton
		{
			Resource = "launch.create",
			Type = DXButton.ButtonType.Positive,

			Horizontal = LayoutOptions.Center,

			ButtonTapped = OnCreateTapped
		};

		createBtn.Init();
		
		layout.Add( createBtn );

		// 'Use Existing Account'
		loginBtn = new DXButton
		{
			Resource = "launch.login",
			Type = DXButton.ButtonType.Action,

			Horizontal = LayoutOptions.Center,

			ButtonTapped = OnLoginTapped
		};

		loginBtn.Init();
		
		layout.Add( loginBtn );

		Content = layout;
	}

	// Resets button states when returning to screen
	public override void Reset()
	{
		createBtn.Reset();
		loginBtn.Reset();
	}
	
	/* Event Callbacks */

	// Move to create account screen
	private static void OnCreateTapped( object sender )
	{
		Shell.Instance.LaunchStack.Push( new LaunchCreate() );
	}

	// Move to login screen
	private static void OnLoginTapped( object sender )
	{
		Shell.Instance.LaunchStack.Push( new LaunchLogin( null ) );
	}

	// Show help tooltip
	private void OnHelpTapped()
	{
		DXTooltip.Show( helpBtn, "launch.start" );
	}

	/* Layout */

	// All common layout
	public override void UpdateLayout( LayoutType type )
	{
		// Default sizing
		titleLbl.FontSize = 53;
		infoLbl.FontSize = 22;

		helpBtn.Size = 30;
		
		UpdateButtons( 230, 48 );

		// Size specific overrides
		base.UpdateLayout( type );
	}

	// Landscape (4:3)
	protected override void Landscape()
	{
		double width = (DXDevice.GetScreenWd() * 0.50);
		double height = (width * 0.113);
		
		titleImg.SetSize( width, height );
	}

	// Portrait (3:4)
	protected override void Portrait()
	{
		double width = (DXDevice.GetScreenWd() * 0.70);
		double height = (width * 0.113);
		
		titleImg.SetSize( width, height );
	}

	// Mobile landscape
	protected override void MobileLandscape()
	{
		titleImg.WidthRequest = (DXDevice.GetScreenWd() * 0.50);

		titleLbl.FontSize = 30;
		infoLbl.FontSize = 17;
		
		UpdateButtons( 200, 48 );
	}

	// Mobile portrait
	protected override void MobilePortrait()
	{
		titleImg.WidthRequest = (DXDevice.GetScreenWd() * 0.75);

		titleLbl.FontSize = 36;
		infoLbl.FontSize = 18;
		
		UpdateButtons( 200, 48 );
	}

	// Updates sizing for both buttons
	private void UpdateButtons( double wd, double ht )
	{
		createBtn.WidthRequest = wd;
		createBtn.HeightRequest = ht;

		loginBtn.WidthRequest = wd;
		loginBtn.HeightRequest = ht;
	}
}

//
