﻿/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using DXLib.UI;
using DXLib.UI.Layout;

using DXLib.UI.Form;
using DXLib.UI.Form.Control;
using DXLib.UI.Form.Message;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Displays data entry form centered within create account screen. User must enter all fields required for account
 * creation here. 
 */
public class LaunchCreateForm : DXGridLayout
{
	/* Properties */
	public string Username => username.Text;
	public string Password => password1.Text;
	public string FirstName => first.Text;
	public string LastName => last.Text;
	public string Role => role.GetObject() as string;

	/* Fields */
	private readonly List<DXFormControl> controls;

	private readonly DXVerticalLayout list1;
	private readonly DXVerticalLayout list2;

	// List 1
	private readonly DXTextField username;
	private readonly DXTextField password1;
	private readonly DXTextField password2;

	// List 2
	private readonly DXTextField first;
	private readonly DXTextField last;
	private readonly DXSelectorField role;

	/* Methods */
	public LaunchCreateForm()
	{
		BackgroundColor = DXColors.Light4;

		// Spacing
		RowSpacing = 16;
		ColumnSpacing = 40;

		// Allocate internal list
		controls = [];

		/* List 1 */

		list1 = new DXVerticalLayout
		{
			Spacing = 16
		};

		// Username
		username = new DXTextField
		{
			Key = "username",
			Title = "create.username",
			Text = null,
			MinLength = 3,
			MaxLength = 64,
			Type = DXTextField.TextType.Email,
			Hint = DXFormControl.HintType.Email,
			Help = "create.user",
		};

		Add( list1, username );

		// Password
		password1 = new DXTextField
		{
			Key = "password",
			Title = "create.password",
			Text = null,
			MinLength = 8,
			MaxLength = 32,
			Type = DXTextField.TextType.Password,
			Hint = DXFormControl.HintType.RequiredRange,
			Help = "create.pswd"
		};

		Add( list1, password1 );

		// Repeat password
		password2 = new DXTextField
		{
			Key = "password2",
			Title = "create.password2",
			Text = null,
			MinLength = 8,
			MaxLength = 32,
			Type = DXTextField.TextType.Password,
			Hint = DXFormControl.HintType.Password,
			Help = "create.pswd2"
		};

		Add( list1, password2 );

		/* List 2 */

		list2 = new DXVerticalLayout
		{
			Spacing = 16
		};

		// First Name
		first = new DXTextField
		{
			Key = "first",
			Title = "create.first",
			Text = null,
			MinLength = 1,
			MaxLength = 32,
			Type = DXTextField.TextType.CapitalizeWords,
			Hint = DXFormControl.HintType.Required,
			Help = "create.first"
		};

		Add( list2, first );

		// Last Name
		last = new DXTextField
		{
			Key = "last",
			Title = "create.last",
			Text = null,
			MinLength = 1,
			MaxLength = 32,
			Type = DXTextField.TextType.CapitalizeWords,
			Hint = DXFormControl.HintType.Required,
			Help = "create.last"
		};

		Add( list2, last );

		// Role
		role = new DXSelectorField
		{
			Key = "role",
			Title = "create.role",
			Items = "user.role",
			SelectedItem = null,
			Hint = DXFormControl.HintType.Required,
			Help = "create.role"
		};

		Add( list2, role );
	}

	// Used internally to configure controls being added to form
	private void Add( DXVerticalLayout list, DXFormControl control )
	{
		control.IsRequired = true;
		control.SetState( DXFormControl.ControlState.Normal );

		control.Init();

		controls.Add( control );
		list.Add( control );
	}

	// Determines if all fields valid, password fields must also match
	public bool IsValid()
	{
		return controls.All( control => control.IsValid() ) && PasswordsMatch();
	}

	// Used internally to determine if both password fields match
	private bool PasswordsMatch()
	{
		return (password2.Text == password1.Text);
	}

	// Shows error for Password2 field if passwords do not match
	public void ValidatePasswords( DXFormControl control )
	{
		bool pswd2 = (control.Key == "password2");
		
		if ( PasswordsMatch() )
		{
			password2.ForceInvalid = false;
			password2.SetState( pswd2 ? DXFormControl.ControlState.Focused : DXFormControl.ControlState.Normal );
		}
		else
		{
			password2.ForceInvalid = true;
			password2.Hint = DXFormControl.HintType.Password;
			password2.SetState( DXFormControl.ControlState.Error );
		}
	}

	// Hides native keyboard 
	public async Task HideKeyboard()
	{
		await username.HideKeyboard();
		
		await password1.HideKeyboard();
		await password2.HideKeyboard();
		
		await first.HideKeyboard();
		await last.HideKeyboard();
	}
	
	/* Layout */

	// Layout based on device size/orientation
	public override void UpdateLayout( LayoutType type )
	{
		ClearAll();

		base.UpdateLayout( type );
	}

	// Landscape
	protected override void Landscape()
	{
		Padding = 36;

		AddStarColumn();
		AddStarColumn();

		Add( list1, 0, 0 );
		Add( list2, 1, 0 );
	}

	// Portrait
	protected override void Portrait()
	{
		Padding = 36;

		AddStarRow();
		AddStarRow();

		Add( list1, 0, 0 );
		Add( list2, 0, 1 );
	}

	// Mobile landscape
	protected override void MobileLandscape()
	{
		Padding = 36;

		AddStarColumn();
		AddStarColumn();

		Add( list1, 0, 0 );
		Add( list2, 1, 0 );
	}
	
	// Mobile portrait
	protected override void MobilePortrait()
	{
		Padding = new Thickness( 20, 32 );

		AddStarRow();
		AddStarRow();

		Add( list1, 0, 0 );
		Add( list2, 0, 1 );
	}
}

//
