﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Container;
using DXLib.UI.Control.Checkbox;

using DXLib.Utils;

namespace iStatVball3;

/*
 * A sub-screen of Settings used to turn on/off RallyFlow Custom Options. Two checkbox lists are provided, one for the
 * primary team and one for the opponent. Can be reused for QuickSelect, Ratings, Modifiers, and Faults.
 */
public class SettingsContentCustom : DXScroll
{
	/* Events */
	public Action<string> SelectionChanged { get; set; }

	/* Properties */
	public string Key { get; private set; }

	// Currently selected options
	public IList<string> Selected { set => node.SelectItems( value ); }
	public IList<string> SelectedOpp { set => nodeOpp.SelectItems( value ); }

	/* Fields */
	private readonly DXCheckboxNode node;
	private readonly DXCheckboxNode nodeOpp;

	/* Methods */
	public SettingsContentCustom( string key )
	{
		Key = key;

		BackgroundColor = DXColors.Light4;

		Padding = new Thickness( 15, 10, 15, 10 );
		Orientation = ScrollOrientation.Vertical;

		// Layout
		DXVerticalLayout layout = new()
		{
			Padding = 0,
			Spacing = 5
		};

		bool rating = (key == Settings.RatingKey);
		bool modifier = (key == Settings.ModifierKey);

		// Key list different for ratings (block NA for modifiers)
		string list = rating ? Settings.CustomRating : Settings.CustomAction;

		// Primary Team
		node = new DXCheckboxNode
		{
			SelectionChanged = OnSelectionChanged
		};

		node.SetRoot( "node", DXString.Get( "settings.node" ) );
		node.SetChildItems( list );

		if ( modifier )
		{
			node.RemoveItem( Stats.BlockKey );
		}

		layout.Add( node );

		// Opponent
		nodeOpp = new DXCheckboxNode
		{
			SelectionChanged = OnSelectionChanged
		};

		nodeOpp.SetRoot( "nodeOpp", DXString.Get( "settings.node.opp" ) );
		nodeOpp.SetChildItems( list );

		if ( modifier )
		{
			nodeOpp.RemoveItem( Stats.BlockKey );
		}

		layout.Add( nodeOpp );

		Content = layout;
	}

	// Returns list of currently selected keys for either team
	public List<string> GetSelectedKeys( bool opponent = false )
	{
		DXCheckboxNode list = opponent ? nodeOpp : node;

		// Use appropriate team
		return list.GetSelectedKeys();
	}

	/* Event Callbacks */

	// Callback registered listener on checkbox state change
	private void OnSelectionChanged()
	{
		SelectionChanged?.Invoke( Key );
	}
}

//
