﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.Log;
using DXLib.UI.Layout;
using DXLib.UI.Container;

using DXLib.Video;
using DXLib.Video.Source;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Video player used to show help topic videos in a modal popup.
 */
public class HelpPlayer
{
	/* Constants */

	// Vimeo signature length has changed
	private const int DefaultSigLength = 40;

	/* Events */
	public Action Closed { get; set; }

	/* Fields */
	private readonly DXVideo player;
	private readonly DXPopup popup;

	// Selected video
	private HelpTreeData currentData;

	/* Methods */
	public HelpPlayer()
	{
		// Required parent layout
		DXGridLayout layout = new()
		{
			Horizontal = LayoutOptions.Fill,
			Vertical = LayoutOptions.Fill
		};
		
		// Reusable video player
		player = new DXVideo
		{
			AutoPlay = false,
			IsLooping = false,
			ShowControls = true,
			
			Horizontal = LayoutOptions.Fill,
			Vertical = LayoutOptions.Fill
		};

		player.UpdateStatus += OnStatusUpdated;

		layout.Add( player, 0, 0 );
		
		// Reusable modal popup
		popup = new DXPopup( layout )
		{
			PopupClosed = OnClosed
		};

		// Close video on orientation change
		DeviceDisplay.MainDisplayInfoChanged += ( sender, args ) =>
		{
			popup.Hide();
		};
	}

	// Show video player for specified video
	public void Show( HelpTreeData data )
	{
		currentData = data;

		// Dynamic video URI
		string videoId = data.VideoId;
		string videoSig = data.VideoSig;

		// Vimeo URI format has changed
		player.Source = DXVideoSource.FromUri( videoSig.Length > DefaultSigLength ? $"https://player.vimeo.com/progressive_redirect/playback/{videoId}/rendition/1080p/file.mp4?loc=external&log_user=0&signature={videoSig}" 
																				  : $"https://player.vimeo.com/external/{videoId}.hd.mp4?s={videoSig}&profile_id=175" );
		// Popup has topic title
		popup.SetTitle( data.Name );

		Size size = GetVideoSize();

		// Dynamically sized
		popup.ViewWidth = size.Width;
		popup.ViewHeight = size.Height;

		popup.Show();

		// In case autoplay not working
		player.Play();
	}

	// Returns dynamic video size based on device platform and orientation
	private static Size GetVideoSize()
	{
		double screenWd = DXDevice.GetScreenWd();
		const double ratio = (4.0 / 3.0);

		bool landscape = DXDevice.IsLandscape();

		double wd, ht;

		// Tablet
		if ( DXDevice.IsTablet )
		{
			wd = (screenWd * (landscape ? 0.75 : 0.90));
			ht = (wd / ratio);
		}
		// Mobile
		else
		{
			if ( landscape )
			{
				ht = (DXDevice.GetScreenHt() * 0.95);
				wd = (ht * ratio);
			}
			else
			{
				wd = (screenWd * 0.95);
				ht = (wd / ratio);
			}
		}

		return new Size( wd, ht );
	}

	/* Event Callbacks */

	// Native video player status was updated
	private async void OnStatusUpdated( object sender, EventArgs args )
	{
		try
		{
			// Mark item as viewed once video ends
			if ( (player.Status == DXVideo.VideoStatus.Ended) && (currentData != null) )
			{
				currentData.Viewed = true;
		
				// Persist viewed status
				await Shell.CurrentUser.UpdateHelp( currentData.Key );
			
				// Dismiss, callback listener
				popup.Hide();

				currentData = null;
			}
		}
		catch ( Exception ex )
		{
			DXLog.Exception( "help.status", ex );
		}
	}
	
	// Video window was dismissed, callback listener
	private void OnClosed()
	{
		player.Stop();
		
		Closed?.Invoke();
	}
}

//
