﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;

using DXLib.UI.Alert;
using DXLib.UI.Layout;
using DXLib.UI.Container;
using DXLib.UI.Control.Button;

using DXLib.Log;
using DXLib.Email;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Implements layout for help drawer footer area, which contains various help links including email, web, and social media. 
 */ 
public class HelpFooter : DXGridLayout
{
	/* Fields */
	private readonly DXIconButton contactIcn;
	private readonly DXTextButton contactTxt;
	
	private readonly DXIconButton subscribeIcn;
	private readonly DXTextButton subscribeTxt;
	
	private readonly DXIconButton webIcn;
	private readonly DXTextButton webTxt;
	
	private readonly DXIconButton logIcn;
	private readonly DXTextButton logTxt;
	
	private readonly DXIconButton facebookIcn;
	private readonly DXTextButton facebookTxt;
	
	private readonly DXIconButton youtubeIcn;
	private readonly DXTextButton youtubeTxt;
	
	/* Methods */
	public HelpFooter()
	{
		IgnoreSafeArea = true;
		BackgroundColor = DXDrawer.FillColor;

		double safeBottom = DXDevice.SafeArea().Bottom;
		
		// Layout
		Padding = (safeBottom > 0) ? new Thickness( 15, 15, 15, safeBottom ) : new Thickness( 15, 10, 15, 10 );
		Margin = 0;

		Horizontal = LayoutOptions.Fill;
		Vertical = LayoutOptions.Fill;
		
		RowSpacing = 10;
		ColumnSpacing = 8;

		const double iconSize = 26;
		const double iconSize2 = 36;

		const double fontSize = 15;
		const double fontSize2 = 11;

		Thickness iconMargin = new( 0, 0, 0, 0 );

		// 4 rows
		AddStarRow();					// 0: contact
		AddStarRow();					// 1: subscribe
		AddStarRow();					// 2: web link
		AddStarRow();					// 3: send log

		// 3 Columns
		AddFixedColumn( iconSize );		// 0: icon
		AddStarColumn();				// 1: label
		AddAutoColumn();				// 2: social

		// Contact us
		contactIcn = new DXIconButton
		{
			Margin = iconMargin,

			Resource = "email",
			Size = iconSize,
			IconColor = DXColors.Action,

			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center,
			
			IsSticky = true,
			ButtonTapped = OnContactTapped
		};

		contactIcn.Init();
		
		Add( contactIcn, 0, 0 );

		contactTxt = new DXTextButton
		{
			Resource = "help.contact",

			Color = DXColors.Action,
			FontSize = fontSize,

			Horizontal = LayoutOptions.Start,
			Vertical = LayoutOptions.Center,

			IsSticky = true,
			ButtonTapped = OnContactTapped
		};

		Add( contactTxt, 1, 0 );

		// Subscribe to newsletter
		subscribeIcn = new DXIconButton
		{
			Margin = iconMargin,

			Resource = "news",
			Size = iconSize,
			IconColor = DXColors.Action,

			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center,

			IsSticky = true,
			ButtonTapped = OnSubscribeTapped
		};

		subscribeIcn.Init();
		
		Add( subscribeIcn, 0, 1 );

		subscribeTxt = new DXTextButton
		{
			Resource = "help.subscribe",

			Color = DXColors.Action,
			FontSize = fontSize,
			
			Horizontal = LayoutOptions.Start,
			Vertical = LayoutOptions.Center,

			IsSticky = true,
			ButtonTapped = OnSubscribeTapped
		};

		Add( subscribeTxt, 1, 1 );

		// Web link
		webIcn = new DXIconButton
		{
			Margin = iconMargin,

			Resource = "web",
			Size = iconSize,
			IconColor = DXColors.Action,

			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center,

			IsSticky = true,
			ButtonTapped = OnWebTapped
		};

		webIcn.Init();
		
		Add( webIcn, 0, 2 );

		webTxt = new DXTextButton
		{
			Resource = "help.web",

			Color = DXColors.Action,
			FontSize = fontSize,
			
			Horizontal = LayoutOptions.Start,
			Vertical = LayoutOptions.Center,

			IsSticky = true,
			ButtonTapped = OnWebTapped
		};

		Add( webTxt, 1, 2 );

        // Log
        logIcn = new DXIconButton
        {
            Margin = iconMargin,

            Resource = "warn",
            Size = iconSize,
            IconColor = DXColors.Action,

            Horizontal = LayoutOptions.Center,
            Vertical = LayoutOptions.Center,

            IsSticky = true,
            ButtonTapped = OnLogTapped
        };

        logIcn.Init();
        
        Add( logIcn, 0, 3 );

        logTxt = new DXTextButton
        {
            Resource = "help.log",

            Color = DXColors.Action,
            FontSize = fontSize,

            Horizontal = LayoutOptions.Start,
            Vertical = LayoutOptions.Center,

            IsSticky = true,
            ButtonTapped = OnLogTapped
        };

        Add( logTxt, 1, 3 );

        // Right-hand area layout
        DXGridLayout rightLayout = new()
		{
			Padding = 0,
			RowSpacing = 0,
			
			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center
		};

		// Facebook
		facebookIcn = new DXIconButton
		{
			Resource = "facebook",
			
			IconColor = DXColors.Dark4,
			IconScale = 0.9,
			
			Size = iconSize2,
			Margin = 0,
			
			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center,

			IsSticky = true,
			ButtonTapped = OnFacebookTapped
		};

		facebookIcn.Init();
		
		rightLayout.AddFixedRow( iconSize2 );
		rightLayout.Add( facebookIcn, 0, 0 );

		facebookTxt = new DXTextButton
		{
			Margin = new Thickness( 0, 2, 0, 0 ),

			Resource = "help.facebook",
			Color = DXColors.Dark4,
			FontSize = fontSize2,

			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center,
			
			IsSticky = true,
			ButtonTapped = OnFacebookTapped
		};

		rightLayout.AddFixedRow( fontSize2 + 3 );
		rightLayout.Add( facebookTxt, 0, 1 );

		// Spacer
		rightLayout.AddFixedRow( 8 );

		// YouTube
		youtubeIcn = new DXIconButton
		{
			Resource = "youtube",

			IconColor = DXColors.Dark4,
			IconScale = 1.0,
			
			Size = iconSize2,
			Margin = 0,

			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center,

			IsSticky = true,
			ButtonTapped = OnYouTubeTapped
		};

		youtubeIcn.Init();
		
		rightLayout.AddFixedRow( iconSize2 );
		rightLayout.Add( youtubeIcn, 0, 3 );

		youtubeTxt = new DXTextButton
		{
			Margin = new Thickness( 0, -3, 0, 0 ),

			Resource = "help.youtube",
			Color = DXColors.Dark4,
			FontSize = fontSize2,
			
			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center,
			
			IsSticky = true,
			ButtonTapped = OnYouTubeTapped
		};

		rightLayout.AddFixedRow( fontSize2 + 3 );
		rightLayout.Add( youtubeTxt, 0, 4 );

		Add( rightLayout, 2, 0, 1, 4 );
	}

	/* Event Callbacks */

	// CONTACT

	// Icon
	private void OnContactTapped()
	{
		contactTxt.ShowTap();
		
		OpenContact();
	}

	// Text
	private void OnContactTapped( object data )
	{
		contactIcn.ShowTap();
		
		OpenContact();
	}

	// Action
	private async void OpenContact()
	{
		EmailMessage msg = new()
		{
			To = [ "support@istatvball.com" ]
		};

		// Open email client
		try
		{
			await Email.ComposeAsync( msg );
			
			ResetContact();
		}
		catch ( FeatureNotSupportedException )
		{
			DXAlert.ShowError( "help.err.email", ResetContact );
		}
		catch ( Exception )
		{
			DXAlert.ShowError( "help.err", ResetContact );
		}
	}

	// Reset
	private void ResetContact()
	{
		contactIcn.Reset();
		contactTxt.Reset();
	}
	
	// SUBSCRIBE

	// Icon
	private void OnSubscribeTapped()
	{
		subscribeTxt.ShowTap();
		
		OpenSubscribe();
	}

	// Text
	private void OnSubscribeTapped( object data )
	{
		subscribeTxt.ShowTap();
		
		OpenSubscribe();
	}

	// Action
	private async void OpenSubscribe()
	{
		Uri uri = new( $"https://cdn.forms-content.sg-form.com/3c5a6e87-7dd2-11eb-8bc8-fe09111ed14e" );

		// Open in external browser
		await Browser.OpenAsync( uri, BrowserLaunchMode.SystemPreferred );

		ResetSubscribe();
	}
	
	// Reset
	private void ResetSubscribe()
	{
		subscribeIcn.Reset();
		subscribeTxt.Reset();
	}

	// WEBSITE

	// Icon
	private void OnWebTapped()
	{
		webTxt.ShowTap();
		
		OpenWeb();
	}

	// Text
	private void OnWebTapped( object data )
	{
		webIcn.ShowTap();
		
		OpenWeb();
	}

	// Action
	private async void OpenWeb()
	{
		Uri uri = new( $"https://www.istatvball.com/support" );

		// Open in external browser
		await Browser.OpenAsync( uri, BrowserLaunchMode.SystemPreferred );
		
		ResetWeb();
	}

	// Reset
	private void ResetWeb()
	{
		webIcn.Reset();
		webTxt.Reset();
	}

	// LOG

	// Icon
    private async void OnLogTapped()
    {
	    logTxt.ShowTap();
	    
        await SendLog();
    }

    // Text
    private async void OnLogTapped( object data )
    {
	    logIcn.ShowTap();
	    
        await SendLog();
    }

    // Action
    private async Task SendLog()
    {
        // Send troubleshooting log via local email
		DXEmail.Result result = await DXLog.Send();

		switch ( result )
		{
			case DXEmail.Result.ErrorUnsupported: DXAlert.ShowError( "help.err.email", ResetLog ); break;
			case DXEmail.Result.Error: DXAlert.ShowError( "help.err", ResetLog ); break;
			case DXEmail.Result.Success: DXAlert.ShowOk( "help.log.subject", "help.log.success", ResetLog ); break;
			default: break;
		}
		
    }

    // Reset
    private void ResetLog()
    {
	    logIcn.Reset();
	    logTxt.Reset();
    }
    
    // FACEBOOK

    // Icon
    private void OnFacebookTapped()
	{
		facebookTxt.ShowTap();
		
		OpenFacebook();
	}

    // Text
	private void OnFacebookTapped( object data )
	{
		facebookIcn.ShowTap();
		
		OpenFacebook();
	}

	// Action
	private async void OpenFacebook()
	{
		Uri uri = new( $"https://www.facebook.com/istatvball" );

		// Open in external browser
		await Browser.OpenAsync( uri, BrowserLaunchMode.SystemPreferred );
		
		ResetFacebook();
	}

	// Reset
	private void ResetFacebook()
	{
		facebookIcn.Reset();
		facebookTxt.Reset();
	}
	
	// YOUTUBE

	// Icon
	private void OnYouTubeTapped()
	{
		youtubeTxt.ShowTap();
		
		OpenYouTube();
	}

	// Text
	private void OnYouTubeTapped( object data )
	{
		youtubeIcn.ShowTap();
		
		OpenYouTube();
	}

	// Action
	private async void OpenYouTube()
	{
		Uri uri = new ( $"https://www.youtube.com/channel/UCjwCNQgJPfUV5tSqgkOQTag" );

		// Open in external browser
		await Browser.OpenAsync( uri, BrowserLaunchMode.SystemPreferred );
		
		ResetYouTube();
	}

	// Reset
	private void ResetYouTube()
	{
		youtubeIcn.Reset();
		youtubeTxt.Reset();
	}
}

//
