﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Container;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Implements layout for account drawer footer area, which contains links to the Terms & Conditions and Privacy Policy.
 */
public class AccountFooter : DXGridLayout
{
	/* Fields */
	private readonly DXIconButton termsIcn;
	private readonly DXTextButton termsTxt;

	private readonly DXIconButton privacyIcn;
	private readonly DXTextButton privacyTxt;

	/* Methods */
	public AccountFooter()
	{
		IgnoreSafeArea = true;
		BackgroundColor = DXDrawer.FillColor;

		double safeBottom = DXDevice.SafeArea().Bottom;

		// Layout
		Padding = (safeBottom > 0) ? new Thickness( 15, 15, 15, safeBottom ) : new Thickness( 15, 10, 15, 10 );
		Margin = 0;
		
		Horizontal = LayoutOptions.Fill;
		Vertical = LayoutOptions.Fill;

		RowSpacing = 10;
		ColumnSpacing = 8;

		const double iconSize = 26;
		const double textSize = 16;

		// 2 rows
		AddStarRow();				    // 0: terms
		AddStarRow();					// 1: privacy

		// 2 Columns
		AddFixedColumn( iconSize );     // 0: icon
		AddStarColumn();                // 1: label

		// Terms & conditions
		termsIcn = new DXIconButton
		{
			Margin = 0,

			Resource = "terms",
			Size = iconSize,
			IconColor = DXColors.Action,

			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center,

			IsSticky = true,
			ButtonTapped = OnTermsTapped
		};

		Add( termsIcn, 0, 0 );

		termsTxt = new DXTextButton
		{
			Resource = "account.terms",

			TextColor = DXColors.Action,
			Font = DXFonts.Roboto,
			FontSize = textSize,

			Horizontal = LayoutOptions.Start,
			Vertical = LayoutOptions.Center,

			IsSticky = true,
			ButtonTapped = OnTermsTapped
		};

		Add( termsTxt, 1, 0 );

		// Privacy policy
		privacyIcn = new DXIconButton
		{
			Margin = 0,

			Resource = "privacy",
			Size = iconSize,
			IconColor = DXColors.Action,

			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center,

			IsSticky = true,
			ButtonTapped = OnPrivacyTapped
		};

		Add( privacyIcn, 0, 1 );

		privacyTxt = new DXTextButton
		{
			Resource = "account.privacy",

			TextColor = DXColors.Action,
			Font = DXFonts.Roboto,
			FontSize = textSize,

			Horizontal = LayoutOptions.Start,
			Vertical = LayoutOptions.Center,

			IsSticky = true,
			ButtonTapped = OnPrivacyTapped
		};

		Add( privacyTxt, 1, 1 );
	}

	/* Event Callbacks */

	// TERMS
	
	// Icon
	private void OnTermsTapped()
	{
		termsTxt.ShowTap();
		
		OpenTerms(); 
	}

	// Text
	private void OnTermsTapped( object data )
	{
		termsIcn.ShowTap();
		
		OpenTerms();
	}

	// Action
	private async void OpenTerms()
	{
		Uri uri = new ( $"https://www.istatvball.com/terms" );

		// Open in external browser window
		await Browser.OpenAsync( uri, BrowserLaunchMode.SystemPreferred );
		
		ResetTerms();
	}
	
	// Reset
	private void ResetTerms()
	{
		termsIcn.Reset();
		termsTxt.Reset();
	}

	// PRIVACY
	
	// Icon
	private void OnPrivacyTapped()
	{
		privacyTxt.ShowTap();
		
		OpenPrivacy();
	}

	// Text
	private void OnPrivacyTapped( object data )
	{
		privacyIcn.ShowTap();
		
		OpenPrivacy();
	}

	// Action
	private async void OpenPrivacy()
	{
		Uri uri = new( $"https://www.istatvball.com/privacy" );

		// Open in external browser window
		await Browser.OpenAsync( uri, BrowserLaunchMode.SystemPreferred );
		
		ResetPrivacy();
	}
	
	// Reset
	private void ResetPrivacy()
	{
		privacyIcn.Reset();
		privacyTxt.Reset();
	}
}

//
