﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Card;
using DXLib.UI.Control.Badge;

using DXLib.Utils;

namespace iStatVball3;

/*
 * A small sized card layout used on many card pages throughout the app. If the card represents a read-only object, a
 * lock icon will be displayed. 
 */
public class CardSmall : CardBase
{
	// /* Constants */
	public static readonly double CardHt = DXDevice.IsTabletWide ? 220 : 200;
	
	/* Properties */
	public string Badge { set => AddBadge( value ); }
	public Color BadgeColor { set => textBadge.Color = value; }

	public string Number { set => AddNumber( value ); }
	public Color NumberColor { set => numberBadge.Color = value; }

	public int Count { set => AddCounter( value ); }
	public string CountTitle { set => counterBadge.Title = DXString.GetPlural( value, counterBadge.Count ); }

	/* Fields */
	private DXTextBadge textBadge;
	private DXNumberBadge numberBadge;
	private DXCounterBadge counterBadge;

	/* Methods */
	public CardSmall() : this( false, false )
	{}

	// Extended constructor
	public CardSmall( bool small, bool readOnly ) : base( readOnly )
	{
		/* Layout */
		
		// 3 Rows
		layout.AddStarRow( 40 );			    // 0: labels
		layout.AddStarRow( 30 );			    // 1: badge
		layout.AddStarRow( 30 );			    // 2: buttons

		// 3 columns
		if ( small )
		{
			layout.AddStarColumn( 30 );         // 0: count/badge
			layout.AddStarColumn( 50 );         // 1: record
			layout.AddStarColumn( 20 );         // 2: image/icon

			layout.Add( imageArea, 2, 0 );
		}
		else
		{
			layout.AddStarColumn( 26 );         // 0: count/badge
			layout.AddStarColumn( 36 );         // 1: record
			layout.AddStarColumn( 38 );         // 2: image/icon

			layout.Add( imageArea, 2, 0, 1, 2 );
		}

		/* Controls */

		// Labels
		TitleSize = 18;
		SubTitleSize = 13;

		labels.SubTitleLbl.Font = DXFonts.Roboto;

		layout.Add( labels, 0, 0, 2, 1 );

		// Analyze/Edit buttons
		buttons.Margin = new Thickness( 10, 10 );
		buttons.Vertical = LayoutOptions.Center;

		layout.Add( buttons, 0, 2, 2, 1 );

		// Lock
		if ( readOnly || GlobalLock )
		{
			layout.Add( lockBtn, 2, 2 );
		}
	}
	
	// Deactivates card (shows as faded)
	public override void SetDeactivated( bool deactivated )
	{
		base.SetDeactivated( deactivated );

		double opacity = deactivated ? DeactivatedOpacity : 1.0;

		// Deactivates all extended controls
		textBadge?.Opacity = opacity;
		numberBadge?.Opacity = opacity;
		counterBadge?.Opacity = opacity;
	}
	
	/* Sizing */

	// Returns constant card height
	public override double GetViewHt()
	{
		return CardHt;
	}

	/* Accessories */
	
	// Card specific link icon layout
	protected override void SetLinkStatus( Link.StatusType? status )
	{
		base.SetLinkStatus( status );

		// Only layout if necessary
		if ( linkBtn is { IsVisible: true } )
		{
			layout.Add( linkBtn, 2, 2 );
		}
	}

	// Card specific root icon layout
	protected override void SetRootStatus( bool rooted )
	{
		base.SetRootStatus( rooted );

		// Root left of link if present
		if ( rooted && (rootBtn != null) )
		{
			double rightPad = linkBtn is { IsVisible: true } ? (IconSize + (IconPad * 2)) : IconPad;
			rootBtn.Margin = new Thickness( IconPad, IconPad, rightPad, IconPad );

			layout.Add( rootBtn, 2, 2 );
		}
	}

	// Add optional 'win-loss-tie' record label
	public override void UpdateRecord( int won, int lost, int tied )
	{
		base.UpdateRecord( won, lost, tied );

		bool counter = (counterBadge != null);

		// Some small cards (tournament) have both counter and record
		int col = counter ? 1 : 0;
		double margin = counter ? 0 : 10;

		recordLbl.Margin = new Thickness( margin, 0, 0, 0 );
		layout.Add( recordLbl, col, 1 );
	}

	/* Badges */
	
	// Adds optional number badge to layout
	private void AddNumber( string value )
	{
		// Number/counter
		if ( string.IsNullOrEmpty( value ) )
		{
			if ( numberBadge != null )
			{
				layout.Remove( numberBadge );
			}

			numberBadge = null;
		}
		else
		{
			numberBadge = new DXNumberBadge
			{
				Text = value,
				AdjustY = -1,
				
				TextColor = DXColors.Light4,
				Color = imageArea.Color,

				Size = 36,
				FontSize = 18,
				
				Padding = new Thickness( 10, 10, 0, 0 ),
				Margin = 0,
				
				Horizontal = LayoutOptions.Start
			};

			numberBadge.Init();

			layout.Add( numberBadge, 0, 1 );
		}
	}

	// Add optional counter badge to layout
	private void AddCounter( int value )
	{
		if ( value >= 0 )
		{
			counterBadge = new DXCounterBadge
			{
				Count = value,
				IsSmall = true,

				Spacing = 0,
				Margin = new Thickness( 10, 10, 10, 0 ),
				
				WidthRequest = 64,
				
				Horizontal = LayoutOptions.Start,
				Vertical = LayoutOptions.Center
			};

			layout.Add( counterBadge, 0, 1 );
		}
	}

	// Add optional text badge to layout
	private void AddBadge( string value )
	{
		if ( value != null )
		{
			textBadge = new DXTextBadge
			{
				TextUpper = DXString.GetUpper( value ),
				TextColor = DXColors.Light4,
				Color = DXColors.Action,

				Margin = 10
			};

			layout.Add( textBadge, 0, 1 );
		}
	}

	// Returns number of card columns for current device/orientation
	private static int GetColumnCount()
	{
		return DXDevice.GetLayoutType( true ) switch
		{
			LayoutType.Landscape or 
			LayoutType.WideLandscape => 3,
			
			LayoutType.Portrait or 
			LayoutType.WidePortrait => 2,
			
			LayoutType.MobileLandscape => 2,
			LayoutType.MobilePortrait => 1,
			
			_ => 0
		};
	}
	
	// Calculates image area size for small cards
	public static Size GetImageSize( bool small = false )
	{
		// Calc image area size based on col,row layout and card size
		double cols = GetColumnCount();
		double fullWd = (DXDevice.GetScreenWd() - DXDevice.SafeAreaLR().HorizontalThickness);
	
		double width = ((fullWd - (DXCardPage.BasePadding * (cols + 1))) / cols) * (small ? 0.20 : 0.38);
		double height = CardHt * (small ? 0.40 : 0.70);
		
		return new Size( width, height );
	}
}

//
