﻿/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using DXLib.UI.Layout;
using DXLib.UI.Control.Image;

using DXLib.Utils;

namespace iStatVball3;

/*
 * An alternative to DXImageArea that provides a grid of images rather than the single image typically used on UI cards.
 * There are pre-defined grid layouts available that dynamically adjust based on orientation.
 */
public class CardGrid : DXGridLayout
{
	/* Constants */

	// Available layout types
	public enum GridType
	{
		Roster,
		Lineup
	}

	/* Properties */
	public GridType Type { get; set; }

	// Image/icon configuration
	public string DefaultIcon { get; set; }
	public int BorderSize { get; set; }

	public Color IconColor { get; set; }
	public Color ImageColor { get; set; }
	public Color BorderColor { set => BackgroundColor = value; }
	
	/* Fields */
	private readonly List<DXImageArea> images;
	private double iconScale;
	
	// Underlying data
	private List<Player> gridPlayers;
	
	/* Methods */
	public CardGrid()
	{
		Horizontal = LayoutOptions.Fill;
		Vertical = LayoutOptions.Fill;

		IsClippedToBounds = true;
	
		// Allocate container
		images = [];
	}

	// Post-construction initialization
	public void Init()
	{
		Margin = 0;
		Padding = 0;

		// Inner dividers, no outer edge
		RowSpacing = BorderSize;
		ColumnSpacing = BorderSize;
	}

	// Sets underlying data
	public void SetPlayers( List<Player> players )
	{
		gridPlayers = players;

		// Require at least 1 photo to be visible
		IsVisible = players is { Count: > 0 };
	}

	// Convenience method for setting size and icon scale
	public void SetSize( Size size, double scale = 1.0 )
	{
		SetSize( size.Width, size.Height, scale );
	}
	
	// Convenience method for setting size and icon scale
	public void SetSize( double width, double height, double scale = 1.0 )
	{
		WidthRequest = width;
		HeightRequest = height;

		iconScale = scale;
	}

	// Populates grid with player images
	private void Populate()
	{
		images.Clear();

		// Size image to cell
		double width = (WidthRequest - (BorderSize * (ColumnCount + 1))) / ColumnCount;
		double height = (HeightRequest - (BorderSize * (RowCount + 1))) / RowCount;

		// Create all image cells
		for ( int row = 0; row < RowCount; row++ )
		{
			for ( int col = 0; col < ColumnCount; col++ )
			{
				// Configure
				DXImageArea imageArea = new()
				{
					DefaultIcon = DefaultIcon,
					
					Color = ImageColor,
					IconColor = IconColor,
					
					Horizontal = LayoutOptions.Fill,
					Vertical = LayoutOptions.Fill,
					
					IsClippedToBounds = true
				};

				imageArea.SetSize( width, height, iconScale );
				
				images.Add( imageArea );
				Add( imageArea, col, row );
			}
		}

		switch ( Type )
		{
			// Roster populated row-col order
			case GridType.Roster:
			{
				for ( int i = 0; i < images.Count; i++ )
				{
					SetImage( i, i );
				}

				break;
			}
			// Lineup populated in rotation order
			case GridType.Lineup:
			{
				SetImage( 0, 3 );
				SetImage( 1, 2 );
				SetImage( 2, 1 );
				SetImage( 3, 4 );
				SetImage( 4, 5 );
				SetImage( 5, 0 );

				break;
			}
			
			default: break;
		}
	}

	// Sets image at specified index within grid with given player image
	private void SetImage( int gridIndex, int playerIndex )
	{
		images[ gridIndex ].SetImage( gridPlayers?[ playerIndex ]?.ImageUrl );
	}
	
	/* Layout */

	// (Re)create grid layout on init or orientation change
	public override void UpdateLayout( LayoutType type )
	{
		if ( IsVisible )
		{
			ClearAll();

			switch ( Type )
			{
				// Roster 6x2 or 4x3 depending on orientation
				case GridType.Roster:
				{
					if ( DXDevice.IsLandscape() )
					{
						AddStarRows( 3 );
						AddStarColumns( 4 );
					}
					else
					{
						AddStarRows( 2 );
						AddStarColumns( 6 );
					}

					break;
				}
				// Lineup always 3x2
				case GridType.Lineup:
				{
					AddStarRows( 2 );
					AddStarColumns( 3 );
					break;
				}
				
				default: break;
			}

			// Load images
			Populate();
		}
	}
}

//
