﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Alert;
using DXLib.UI.Card;

using DXLib.UI.Control;
using DXLib.UI.Control.Image;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Base class for all card pages cards other than on Season Home. Provides functionality common to all cards.
 */
public abstract class CardBase : DXCard
{
	/* Constants */
	protected const double IconSize = 30;
	protected const double IconPad = 10;

	// Deactivated cards are faded
	protected const double DeactivatedOpacity = 0.6;

	// Non-tappable state
	private static readonly Color NonTappableColor = DXColors.Light2;
	private static readonly Color NonTappableLabelColor = DXColors.Light1;
	
	/* Events */
	public Action<DXCard> AnalyzeTapped { get; set; }
	public Action<DXCard> EditTapped { get; set; }

	/* Properties */
	public string DefaultIcon { set => imageArea.DefaultIcon = value; }
	public string ImageUrl { set => imageArea.SetImage( value ); }
	public virtual Color ImageColor { set => imageArea.Color = value; }

	public string SuperTitle { set => labels.SuperTitle = value; }
	public string Title { set => labels.Title = value; }
	public string SubTitle { set => labels.SubTitle = value; }

	// Font size
	public double SuperTitleSize { set => labels.SuperTitleSize = value; }
	public double TitleSize { set => labels.TitleSize = value; }
	public double SubTitleSize { set => labels.SubTitleSize = value; }

	// Controls link/root badge on card
	public Link.StatusType? LinkStatus { set => SetLinkStatus( value ); }
	public bool RootStatus { set => SetRootStatus( value ); }

	// Controls button availability
	public bool HasAnalyze { set => buttons.HasAnalyze = value; }
	public bool HasEdit { set => buttons.HasEdit = value; }

	public bool IsAnalyzeEnabled { set => buttons.IsAnalyzeEnabled = value; }
	public bool IsEditEnabled { set => buttons.IsEditEnabled = value; }

	/* Field */
	protected readonly DXImageArea imageArea;
	protected readonly DXCardLabels labels;
	protected readonly DXLabel recordLbl;

	protected readonly DXIconButton lockBtn;
	protected DXIconButton linkBtn;
	protected DXIconButton rootBtn;

	protected readonly CardButtons buttons;

	/* Methods */
	protected CardBase( bool readOnly )
	{
		Padding = 0;
		Margin = 0;

		// Icon/photo
		imageArea = new DXImageArea
		{
			Padding = 0,
			Margin = 0,
			
			Horizontal = LayoutOptions.Fill,
			Vertical = LayoutOptions.Fill
		};
		
		// Title/Subtitle
		labels = [];

		// Record
		recordLbl = new DXLabel
		{
			TextColor = DXColors.Dark4,
			
			Font = DXFonts.Oswald,
			FontSize = 28,

			Padding = new Thickness( 0, 5, 0, 0 ),
			Margin = 0,
			
			WidthRequest = 150,
			
			Horizontal = LayoutOptions.Start,
			Vertical = LayoutOptions.Center
		};

		// Analyze/Edit
		buttons = new CardButtons
		{
			AnalyzeTapped = OnAnalyzeTapped,
			EditTapped = OnEditTapped
		};

		// Lock button (read-only)
		if ( readOnly || GlobalLock )
		{
			lockBtn = new DXIconButton
			{
				Resource = "lock_solid",
				IconColor = DXColors.Dark4,

				Size = IconSize,
				Margin = new Thickness( 0, 0, IconPad, IconPad ),

				Horizontal = LayoutOptions.End,
				Vertical = LayoutOptions.End,

				IsSticky = true,
				ButtonTapped = OnLockTapped
			};
		}
	}

	// Deactivates card showing components as faded
	public virtual void SetDeactivated( bool deactivated )
	{
		double opacity = deactivated ? DeactivatedOpacity : 1.0;

		// Deactivate all base components
		imageArea.Opacity = opacity;
		labels.Opacity = opacity;
		recordLbl.Opacity = opacity;
	}

	// Adds link icon with appropriate status color
	protected virtual void SetLinkStatus( Link.StatusType? status )
	{
		bool visible = status is >= Link.StatusType.Pending;
		bool perm = (Shell.CurrentUser.Level >= User.LevelType.Coach);

		if ( visible && perm )
		{
			// Lazily create
			linkBtn ??= new DXIconButton
			{
				Resource = "status",

				Size = IconSize,
				Margin = IconPad,

				Horizontal = LayoutOptions.End,
				Vertical = LayoutOptions.End,

				IsSticky = true,
				ButtonTapped = OnLinkTapped
			};

			// Indicate status
			linkBtn.IconColor = LinkField.StatusColors[ (int)status ];
		}

		// Hide if not linked or pending
		if ( linkBtn != null )
		{
			linkBtn.IsVisible = visible;
		}
	}

	// Adds root link icon when applicable
	protected virtual void SetRootStatus( bool rooted )
	{
		// Lazily create icon if rooted
		if ( rooted && (rootBtn == null) )
		{
			rootBtn = new DXIconButton
			{
				Resource = "link",
				IconColor = DXColors.Positive,

				Size = IconSize,
				Margin = IconPad,

				Horizontal = LayoutOptions.End,
				Vertical = LayoutOptions.End,

				IsSticky = true,
				ButtonTapped = OnRootTapped
			};
		}

		// Hide if not rooted
		if ( rootBtn != null )
		{
			rootBtn.IsVisible = rooted;
		}
	}

	// Updates 'win-loss-tie' record label
	public virtual void UpdateRecord( int won, int lost, int tied )
	{
		if ( (won + lost + tied) > 0 )
		{
			// 'X-Y'
			string record = $"{won}-{lost}";

			// Only show ties if they exist
			if ( tied > 0 )
			{
				record += $"-{tied}";
			}

			recordLbl.Text = record;
		}
	}

	// Sets background and label area colors based on tappable state
	protected override void SetTappable( bool value )
	{
		base.SetTappable( value );

		BackgroundColor = tappable ? DXColors.Light4 : NonTappableColor;
		labels.BackgroundColor = tappable ? DXColors.Light2 : NonTappableLabelColor;
	}

	// Used to exclude buttons from card tap area
	protected override bool IsChildBounds( Point point )
	{
		return buttons.IsChildBounds( point ) || ((lockBtn != null) && lockBtn.Contains( point ));
	}

	// Resets button sticky states
	public override void Reset()
	{
		base.Reset();

		buttons.Reset();
	}

	// Allows concrete card to customize root button tooltip
	protected virtual Task<string> GetRootText()
	{
		return null;
	}

	/* Event Callbacks */

	// Analyzed button tapped
	private void OnAnalyzeTapped()
	{
		AnalyzeTapped?.Invoke( this );
	}

	// Edit button tapped
	private void OnEditTapped()
	{
		EditTapped?.Invoke( this );
	}

	// Shows tooltip message when read-only lock tapped
	private void OnLockTapped()
	{
		DXTooltip.Show( lockBtn, "form.lock" );
	}

	// Shows tooltip message when link tapped
	private void OnLinkTapped()
	{
		DXTooltip.Show( linkBtn, "card.link" );
	}

	// Shows tooltip message when root tapped
	private async void OnRootTapped()
	{
		string text = await GetRootText();

		// Link deleted
		if ( text == null )
		{
			DXAlert.ShowError( "root.link", "root.link.err" );

			rootBtn.IsDisabled = true;
        }
		// Valid
		else
		{
			DXTooltip.ShowText( rootBtn, text );
		}
	}
	
	/* Layout */

	// Device/orientation specific layout
	public override void UpdateLayout( LayoutType type )
	{
		base.UpdateLayout( type );
		
		// Adjust icon/image sizing
		imageArea.UpdateLayout( type );
	}
}

//
