﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Container;

using DXLib.UI.Control;
using DXLib.UI.Control.Image;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace iStatVball3;

/*
 * App navigation bar displayed at top of main UI. Includes back button, logo, screen title, toolbar, etc.
 */
public class AppBar : DXContent
{
	/* Constants */
	public static readonly Color DefaultColor = DXColors.Dark2;

	// Sizing
	private static readonly double BaseHt = DXDevice.IsTablet ? 34 : 24;
	private const double BackHt = 28;

	private static readonly Thickness BasePad = new( 15, 11 );
	public static readonly double ViewHt = (BaseHt + BasePad.VerticalThickness);

	/* Events */
	public Action BackTapped { get; set; }

	/* Properties */
	public bool HasBackButton { get => hasBackButton; set { hasBackButton = value; backBtn.IsVisible = value; } }
	public bool HasToolBar { get => hasToolBar; set { hasToolBar = value; ToolBar.IsVisible = value; } }

	public new bool IsVisible { get => base.IsVisible; set => base.IsVisible = value; }
	public bool IsBackEnabled { set => backBtn.IsVisible = (hasBackButton && value); }

	// Screen title
	public string Title { set => SetTitle ( value ); }
	public string Subtitle { set => SetSubtitle( value ); }

	// Toolbar
	public ToolBar ToolBar { get; }

	/* Fields */
	private readonly DXGridLayout layout;

	private readonly DXIconButton backBtn;
	private readonly DXImage logo;

	private readonly DXLabel titleLbl;
	private readonly DXLabel subtitleLbl;

	// State
	private bool hasBackButton;
	private bool hasToolBar;

	// Titles
	private string title;
	private string subtitle;

	/* Methods */
	public AppBar()
	{
		bool ios = DXDevice.IsIOS;
		bool tablet = DXDevice.IsTablet;
		bool wide = DXDevice.IsTabletWide;

		BackgroundColor = DefaultColor;

		Margin = 0;
		Padding = 0;
		
		// Grid
		layout = new DXGridLayout
		{
			BackgroundColor = DefaultColor,

			Margin = 0,
			Padding = 0,			// Set dynamically
			RowSpacing = 0,
			
			IgnoreSafeArea = true,
			
			Horizontal = LayoutOptions.Fill,
			Vertical = LayoutOptions.Fill
		};

		// Back
		backBtn = new DXIconButton
		{
			Resource = ios ? "back_ios" : "back",
			
			Size = BackHt,
			IconColor = DXColors.Light2,

			Padding = ToolBar.Pad,
			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center,
			
			ButtonTapped = OnBackTapped
		};

		double logoWd = tablet ? 68 : 54;
		double logoHt = tablet ? 32 : 25;

		// Logo
		logo = new DXImage
		{
			DownsampleToViewSize = false,
			
			WidthRequest = logoWd,
			HeightRequest = logoHt,

			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center
		};

		logo.FromResource( "logo_appbar.png" );

		// Title
		titleLbl = new DXLabel
		{
			TextColor = DXColors.Light2,
			FontSize = 22,

			Margin = new Thickness( 0, (ios ? 0 : -4), 0, 0 ),
			VAlign = TextAlignment.Center,

			Horizontal = LayoutOptions.Start,
			Vertical = LayoutOptions.Center
		};

		// Subtitle
		subtitleLbl = new DXLabel
		{
			TextColor = DXColors.Light2,
			FontSize = 22,

			Margin = new Thickness( 0, (ios ? 1 : (wide ? -5 : -3)), 0, 0 ),
			VAlign = TextAlignment.Center,

			Horizontal = LayoutOptions.Start,
			Vertical = LayoutOptions.Center
		};

		// Toolbar
		ToolBar = new ToolBar( true );

		// Drop shadow
		Shadow = new Shadow()
		{
			Brush = Colors.Black,
			Opacity = 0.9f,
			Offset = new Point( 0, 2 ),
			Radius = 3
		};
		
		// Initial layout
		UpdateLayout();

		Content = layout;
	}

	// Updates title text
	private void SetTitle( string value )
	{
		title = value;
		titleLbl.Text = value;
	}

	// Updates subtitle text
	private void SetSubtitle( string value )
	{
		subtitle = value;
		subtitleLbl.Text = value;
	}

	// Resets back button tap state
	public void ResetBack()
	{
		backBtn.Reset();
	}
	
	/* Event Callbacks */

	// Back button tapped, callback listener
	private void OnBackTapped()
	{
		BackTapped?.Invoke();
	}

	/* Layout */

	// Updates device/orientation specific layout
	public override void UpdateLayout( LayoutType type )
	{
		// Must rebuild every time
		layout.ClearAll();

		// Update title/subtitle
		SetFonts( type );

		// Size specific layouts
		base.UpdateLayout( type );

		// Update children (must be last)
		ToolBar.UpdateLayout( type );
	}

	// Landscape (4:3)
	protected override void Landscape()
	{
		layout.Padding = new Thickness( 0, 5, 0, 0);
		layout.ColumnSpacing = 16;

		// 5 columns
		layout.AddFixedColumn( BackHt );				// 0: Back
		layout.AddFixedColumn( 68 );  					// 1: Logo
		layout.AddAutoColumn();							// 2: Title
		layout.AddStarColumn();							// 3: Subtitle
		layout.AddFixedColumn( ToolBar.GetWidth() );	// 4: Toolbar

		AddChildren();
	}

	// Portrait (3:4)
	protected override void Portrait()
	{
		layout.Padding = new Thickness( 0, 5, 0, 0);
		layout.ColumnSpacing = 16;

		// 5 columns
		layout.AddFixedColumn( BackHt );				// 0: Back
		layout.AddFixedColumn( 68 );					// 1: Logo
		layout.AddAutoColumn();							// 2: Title
		layout.AddStarColumn();    						// 3: Subtitle
		layout.AddFixedColumn( ToolBar.GetWidth() );	// 4: Toolbar

		AddChildren();
	}

	// Mobile Landscape
	protected override void MobileLandscape()
	{
		ToolBar.Mode = ToolBar.DisplayMode.OneLine;

		layout.Padding = DXUtils.AddPadding( BasePad, DXDevice.SafeAreaLTR() );
		layout.ColumnSpacing = 16;

		// 5 columns
		layout.AddFixedColumn( ToolBar.IconSize );		// 0: Back
		layout.AddFixedColumn( 54 );					// 1: Logo
		layout.AddAutoColumn();							// 2: Title
		layout.AddStarColumn();							// 3: Subtitle
		layout.AddFixedColumn( ToolBar.GetWidth() );	// 4: Toolbar

		AddChildren();
	}

	// Mobile portrait
	protected override void MobilePortrait()
	{
		ToolBar.Mode = ToolBar.DisplayMode.More;

		// No extra base pad on top
		layout.Padding = new Thickness( BasePad.Left, DXDevice.GetSafeTop(), BasePad.Right, BasePad.Bottom );
		layout.ColumnSpacing = 12;

		// 5 columns
		layout.AddFixedColumn( ToolBar.IconSize );		// 0: Back
		layout.AddFixedColumn( 54 );					// 1: Logo
		layout.AddAutoColumn();							// 2: Title
		layout.AddStarColumn();							// 3: Subtitle
		layout.AddFixedColumn( ToolBar.GetWidth() );    // 4: Toolbar

		AddChildren();
	}

	// Rebuild grid
	private void AddChildren()
	{
		layout.Add( backBtn, 0, 0 );
		layout.Add( logo, 1, 0 );
		layout.Add( titleLbl, 2, 0 );
		layout.Add( subtitleLbl, 3, 0 );
		layout.Add( ToolBar, 4, 0 );
	}

	// Sets title/subtitle fonts given layout orientation
	private void SetFonts( LayoutType type )
	{
		bool hasSubtitle = (subtitle != null);

		// Condensed (no subtitle)
		if ( type == LayoutType.MobilePortrait )
		{
			titleLbl.Text = hasSubtitle ? subtitle : title;
			titleLbl.Font = DXFonts.RobotoCondensed;

			subtitleLbl.IsVisible = false;
		}
		// Normal
		else
		{
			titleLbl.Text = title;
			titleLbl.Font = hasSubtitle ? DXFonts.RobotoBold : DXFonts.Roboto;

			subtitleLbl.Font = DXFonts.Roboto;
			subtitleLbl.IsVisible = hasSubtitle;
		}
	}
}

//
