﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Control;
using DXLib.UI.Container;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Tappable wrapper for a ScoreCard together with title label.
 */
public sealed class ScoreButton : DXContent
{
	/* Constants */
	private const int TapDelay = 250;
	
	/* Events */
	public Action ScoreTapped { get; set; }
	public Action ScorePressed { get; set; }

	/* Properties */
	public string Team { set => teamLbl.Text = value.ToUpper(); }
	public double TeamSize { get => teamLbl.FontSize; set => teamLbl.FontSize = value; }

	public int Score { get => card.Score; set => card.Score = value; }
	public double ScoreSize { set => card.ScoreSize = value; }

	public bool IsDisabled { set => SetDisabled( value ); }
	public bool ShowDisabled { set => card.ShowDisabled = value; }

	public bool HasShadow { set => card.HasShadow = value; }

	// Team color bar
	public Color BarColor { set => card.BarColor = value; }
	public double BarHt { set => card.BarHt = (float)value; }

	/* Fields */
	private readonly DXLabel teamLbl;
	private readonly ScoreCard card;

	private readonly DXAbsoluteLayout layout;

	// State control
	private bool isDisabled;

	/* Methods */
	public ScoreButton( DXAbsoluteLayout layout )
	{
		this.layout = layout;

		Horizontal = LayoutOptions.Center;
		Vertical = LayoutOptions.Fill;
		
		// Team name
		teamLbl = new DXLabel
		{
			Font = DXFonts.RobotoBoldItalic,

			AdjustY = DXDevice.IsTabletWide ? -7 : -2,
			HAlign = TextAlignment.Center
		};

		layout.Add( teamLbl );

		// Scorecard
		card = new ScoreCard
		{
			CardColor = DXColors.Light4,
			CornerRadius = 4,

			HasShadow = true,
			HasOutline = false,
			
			Tapped = OnTapped,
			Pressed = OnPressed
		};

		layout.Add( card );

		Reset();

		// Register for events
		layout.Add( this );

		// Defaults
		BarHt = 14;
	}

	// Resets button sticky state
	public void Reset()
	{
		teamLbl.TextColor = DXColors.Light4;

		card.ScoreColor = DXColors.Dark1;
		card.CardColor = DXColors.Light4;
		card.HasShadow = true;

		card.Redraw();
	}

	// Draws button in sticky state
	private void DrawSticky()
	{
		teamLbl.TextColor = DXColors.Light1;

		card.ScoreColor = DXColors.Dark3;
		card.CardColor = DXColors.Light1;
		card.HasShadow = false;

		card.Redraw();
	}

	// Sets tap enabled state for this button
	private void SetDisabled( bool disabled )
	{
		Reset();

		isDisabled = disabled;
		card.IsDisabled = disabled;
	}

	/* Events Callbacks */

	// Used tapped anywhere on card
	private void OnTapped()
	{
		if ( !isDisabled )
		{
			// Draw in sticky state
			DrawSticky();

			// Brief tap delay
			DXTimer.Delay( TapDelay, ScoreTapped );
		}
	}

	// User long pressed on card
	private void OnPressed()
	{
		if ( !isDisabled )
		{
			// Always sticky
			DrawSticky();

			// Callback immediately
			ScorePressed?.Invoke();
		}
	}

	/* Layout */

	// Sets draw location of control in parent layout
	public void Layout( double x, double y, double wd, double ht )
	{
		double adjX = (x + 3);
		
		layout.SetBounds( this, adjX, y, wd, ht );

		layout.SetBounds( teamLbl, adjX, y, wd, TeamSize );
		layout.SetBounds( card, adjX, (y + TeamSize + 6), wd, ht );
	}
}

//
