﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Control;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Parent of all icon and number buttons used in the recording UI drawers.
 */ 
public class DrawerBase
{
	/* Constants */
	public static readonly double Spacing = DXDevice.IsIOS ? 5 : 3;

	/* Properties */
	public bool HasTitle => !string.IsNullOrEmpty( Title );

	public string Title { get => titleLbl.Text; set => titleLbl.Text = DXString.Get( value ); }
	public double TitleSize { get => titleLbl.FontSize; set => titleLbl.FontSize = value; }
	public double TitleHt => DXDevice.IsTablet ? (DXDevice.IsIOS? TitleSize : (TitleSize + 2)) : TitleSize;

	public double ChildY => TitleHt + Spacing;

	// Hides entire control
	public bool IsVisible { get => isVisible; set => SetVisible( value ); }

	// Disables entire control
	public bool IsDisabled { get => isDisabled; set => SetDisabled( value ); }

	// Allows title to shift outside of child button bounds
	public double AdjustX { get; set; }

	/* Fields */
	protected readonly DXLabel titleLbl;
	protected readonly DXAbsoluteLayout layout;

	// State control
	protected bool isVisible;
	protected bool isDisabled;

	/* Methods */
	public DrawerBase( DXAbsoluteLayout layout )
	{
		this.layout = layout;

		bool tablet = DXDevice.IsTablet;

		// Title
		titleLbl = new DXLabel
		{
			TextColor = DXColors.Light1,

			Font = DXFonts.Roboto,
			FontSize = tablet ? 12 : 10,

			AdjustY = -3,

			LineBreakMode = LineBreakMode.NoWrap,
			Horizontal = LayoutOptions.Center
		};

		layout.Add( titleLbl );
	}

	// Post construction initialization
	public virtual void Init()
	{}

	// Hides title, children should extend and hide
	protected virtual void SetVisible( bool visible )
	{
		isVisible = visible;

		titleLbl.IsVisible = visible;
	}

	// Visually disables title, children should extend and disable
	protected virtual void SetDisabled( bool disabled )
	{
		isDisabled = disabled;

		titleLbl.Opacity = disabled ? 0.3 : 1.0;
	}

	// Removes UI components from parent view
	public virtual void Remove()
	{
		layout.Remove( titleLbl );
	}

	/* Layout */

	// Sets draw location of control in parent layout
	public virtual void Layout( double x, double y, double size )
	{
		// Account for shape shadow when centering
		if ( HasTitle )
		{
			layout.SetBounds( titleLbl, (x + AdjustX), y, (size - (AdjustX * 2)), TitleHt );
		}
	}
}

//
