﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Implements a UI component for switching between two liberos. 
 */
public class TeamLiberoSwitch : DXGridLayout
{
	/* Constants */
	private const double Radius = 4;
	
	// Colors
	private static readonly Color ActiveColor = DXColors.Action;
	private static readonly Color InactiveColor = DXColors.Dark3;

	/* Events */
	public Action<int> Switched { get; set; }

	/* Properties */

	// Number available liberos, currently active libero number
	public int LiberoCount { set => SetCount( value ); }
	public int Active { set => SetActive( value ); }

	// Enable/disable
	public bool IsDisabled { set => SetDisabled( value ); }

	// Layout control
	public double NumberSize { set => SetNumberSize( value ); }

	/* Fields */
	private readonly DXNumberButton libero1Btn;
	private readonly DXNumberButton libero2Btn;

	// State control
	private int liberoCount;
	private bool isActive1;
	private bool isForcedDisable;

	/* Methods */
	public TeamLiberoSwitch()
	{
		bool ios = DXDevice.IsIOS;
		
		BackgroundColor = DXColors.Dark2;

		Padding = 0;
		Margin = 0;
		
		RowSpacing = 0;
		ColumnSpacing = 0;

		Horizontal = LayoutOptions.Fill;
		Vertical = LayoutOptions.Fill;

		IsClippedToBounds = false;
		IgnoreSafeArea = true;
		
		// Libero 1
		libero1Btn = new DXNumberButton
		{
			NumberColor = DXColors.Light4,
			TextAdjustY =  (ios ? -1 : -4),
			CornerRadius = 0,					// Custom
			
			IsTapShown = false,
			HasShadow = false,
			
			Horizontal = LayoutOptions.Fill,
			Vertical = LayoutOptions.Fill,
			
			ButtonTapped = OnOneTapped
		};

		libero1Btn.Init();
		
		// Libero 2
		libero2Btn = new DXNumberButton
		{
			NumberColor = DXColors.Light4,
			TextAdjustY = (ios ? -1 : -4),
			CornerRadius = 0,					// Custom

			IsTapShown = false,
			HasShadow = false,
			
			Horizontal = LayoutOptions.Fill,
			Vertical = LayoutOptions.Fill,

			ButtonTapped = OnTwoTapped
		};

		libero2Btn.Init();
		
		// Default libero 1
		SetActive( true );
	}

	// Dynamically sets size of entire 1/2 button
	public void SetSwitchSize( double wd, double ht )
	{
		bool tablet = DXDevice.IsTablet;
		bool landscape = DXDevice.IsLandscape();
		
		// Buttons top/bottom (tablet and mobile-portrait)
		if ( tablet || !landscape )
		{
			WidthRequest = wd;
			HeightRequest = ht;

			double halfHt = (ht / 2.0);
			
			libero1Btn.ButtonWd = wd;
			libero2Btn.ButtonWd = wd;
			
			libero1Btn.ButtonHt = halfHt;
			libero2Btn.ButtonHt = halfHt;
		}
		// Buttons left-right (mobile-landscape)
		else
		{
			WidthRequest = wd;
			HeightRequest = ht;

			double halfWd = (wd / 2.0);

			libero1Btn.ButtonWd = halfWd;
			libero2Btn.ButtonWd = halfWd;

			libero1Btn.ButtonHt = ht;
			libero2Btn.ButtonHt = ht;
		}
	}
	
	// Sets currently active libero
	private void SetActive( int number )
	{
		SetActive( number == 1 );
	}

	// Sets currently active libero
	private void SetActive( bool active1 )
	{
		isActive1 = active1;

		DXNumberButton activeBtn = active1 ? libero1Btn : libero2Btn;
		DXNumberButton inactiveBtn = active1 ? libero2Btn : libero1Btn;

		// Update UI
		activeBtn.Color = ActiveColor;
		inactiveBtn.Color = InactiveColor;
	}

	// Switches between libero 1/2
	private void ToggleActive()
	{
		SetActive( !isActive1 );
	}

	// Sets number of available liberos
	private void SetCount( int count )
	{
		liberoCount = count;

		switch ( count )
		{
			// No liberos, neither button shown
			case 0:
			{
				libero1Btn.Text = null;
				libero2Btn.Text = null;

				libero1Btn.Color = InactiveColor;
				libero2Btn.Color = InactiveColor;

				UpdateDisabled( true );
				break;
			}
			// One libero, only first button active
			case 1:
			{
				libero1Btn.Number = 1;
				libero2Btn.Text = null;

				SetActive( isActive1 );
				UpdateDisabled( true );
				break;
			}
			// Two liberos, both buttons active
			case 2:
			{
				libero1Btn.Number = 1;
				libero2Btn.Number = 2;

				SetActive( isActive1 );
				UpdateDisabled( false );
				break;
			}
		}
	}

	// Forces both buttons to be disabled
	private void SetDisabled( bool disabled )
	{
		isForcedDisable = disabled;

		// Both fully disabled
		if ( disabled )
		{
			libero1Btn.IsDisabled = true;
			libero2Btn.IsDisabled = true;
		}
		// Disabled (based on state)
		else
		{
			UpdateDisabled( disabled );
		}
	}

	// Updates disabled status of both buttons based on state
	private void UpdateDisabled( bool disabled )
	{
		bool hasTwo = (liberoCount == 2);

		libero1Btn.IsDisabled = isForcedDisable || !hasTwo || disabled;
		libero2Btn.IsDisabled = isForcedDisable || !hasTwo || disabled;
	}

	// Updates button font size
	private void SetNumberSize( double size )
	{
		libero1Btn.NumberSize = size;
		libero2Btn.NumberSize = size;
	}

	/* Event Callbacks */

	// User tapped first libero button
	private void OnOneTapped( object sender )
	{
		// Switch libero, callback listener
		if ( !isActive1 )
		{
			ToggleActive();

			Switched?.Invoke( 1 );
		}
	}

	// User tapped second libero button
	private void OnTwoTapped( object sender )
	{
		// Switch libero, callback listener
		if ( isActive1 )
		{
			ToggleActive();

			Switched?.Invoke( 2 );
		}
	}

	/* Layout */

	// Rearranges libero 1/2 based on orientation
	public override void UpdateLayout( LayoutType type )
	{
		ClearAll();

		switch ( type )
		{
			// Horizontal
			case LayoutType.WideLandscape:
			case LayoutType.MobileLandscape:
			{
				libero1Btn.UpdateRadius( new CornerRadius( Radius, 0, Radius, 0 ) );
				libero2Btn.UpdateRadius( new CornerRadius( 0, Radius, 0, Radius ) );
				
				AddStarColumn( 50 );
				AddStarColumn( 50 );
		
				Add( libero1Btn, 0, 0 );
				Add( libero2Btn, 1, 0 );
				break;
			}
			// Vertical
			default:
			{
				libero1Btn.UpdateRadius( new CornerRadius( Radius, Radius, 0, 0 ) );
				libero2Btn.UpdateRadius( new CornerRadius( 0, 0, Radius, Radius ) );
				
				AddStarRow( 50 );
				AddStarRow( 50 );
		
				Add( libero1Btn, 0, 0 );
				Add( libero2Btn, 0, 1 );
				break;
			}
		}
	}
}

//
