﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Gestures;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Individual cell representing the libero position in a lineup. The cell can show one of two possible liberos.
 */
public class TeamLibero : DXGridLayout
{
	/* Properties */

	// Are both libero zones empty?
	public bool IsEmpty => (libero1.IsEmpty && libero2.IsEmpty);

	/* Fields */
	private readonly TeamOverlay parent;
	private readonly DXAbsoluteGestures layout;

	// Libero 1/2
	private TeamCell libero1;
	private TeamCell libero2;

	/* Methods */
	public TeamLibero( TeamOverlay parent, DXAbsoluteGestures layout )
	{
		this.parent = parent;
		this.layout = layout;

		IsVisible = false;

		RowSpacing = 0;
		ColumnSpacing = 0;
	}

	// Post-construction initialization
	public void Init()
	{
		BackgroundColor = DXColors.Dark1;

		// Underlying cells
		libero1 = new TeamCell( parent, Lineup.LiberoZone1 );
		libero2 = new TeamCell( parent, Lineup.LiberoZone2 );
	}

	// Determines if libero currently in either libero area
	public bool IsLiberoOut()
	{
		return (!libero1.IsEmpty && libero1.IsLibero) && (libero2.IsEmpty || libero2.IsLibero);
	}

	/* Get/Set */

	// Returns number of liberos currently being used
	public int GetCount()
	{
		int count = 0;

		if ( !libero1.IsEmpty )
		{
			count++;
		}

		if ( !libero2.IsEmpty )
		{
			count++;
		}

		return count;
	}

	// Returns list of liberos selected in UI
	public List<Player> GetPlayers()
	{
		List<Player> list = new( 2 );

		// May be undefined
		if ( !libero1.IsEmpty )
		{
			list.Add( libero1.Player );
		}

		if ( !libero2.IsEmpty )
		{
			list.Add( libero2.Player );
		}

		return list;
	}

	// Returns lineup entry for specified libero area
	public LineupEntry GetEntry( int number )
	{
		return (number == 1) ? libero1.GetEntry() : libero2.GetEntry();
	}

	// Returns lineup entries for both liberos (even if empty)
	public List<LineupEntry> GetEntries()
	{
		return
		[
			// Libero 1
			new LineupEntry
			{
				Zone = Lineup.LiberoZone1,

				Position = libero1.Position,
				IsAltNumber = libero1.IsAltNumber,

				PlayerId = libero1.Player?.UniqueId,
				Player = libero1.Player
			},
			// Libero 2
			new LineupEntry
			{
				Zone = Lineup.LiberoZone2,

				Position = libero2.Position,
				IsAltNumber = libero2.IsAltNumber,

				PlayerId = libero2.Player?.UniqueId,
				Player = libero2.Player
			}
		];
	}

	// Sets lineup entry for specified libero
	public void SetEntry( LineupEntry entry, int number )
	{
		TeamCell cell = (number == 1) ? libero1 : libero2;

		cell.SetEntry( entry );
	}

	// Sets players for both libero areas
	public void SetEntries( List<LineupEntry> entries )
	{
		libero1.SetEntry( entries[ Lineup.LiberoZone1 - 1 ] );
		libero2.SetEntry( entries[ Lineup.LiberoZone2 - 1 ] );
	}

	/* Show/Hide */

	// Displays libero overlay on opposite side from court overlay
	public void Show( bool sideA, Rect court, double teambarSize )
	{
		ClearAll();

		bool ios = DXDevice.IsIOS;
		bool tablet = DXDevice.IsTablet;
		bool wide = DXDevice.IsTabletWide;

		double x, y;
		double wd, ht;

		double statusHt = ios ? DXDevice.GetSafeTop() : 0;

		double scale = CourtView.DpScale;
		double courtSize = (CourtView.CourtSize / scale);
		double net = (CourtView.NetSize / scale);

		// Landscape
		if ( parent.IsLandscape )
		{
			AddStarRows( 2 );

			Add( libero1, 0, 0 );
			Add( libero2, 0, 1 );

			Padding = tablet ? 0 : 2;

			double border = tablet ? 4 : 2;
			double outWd = ((court.Width - (courtSize * 2.0) - net) / 2.0);
			double rowHt = (courtSize / 3.0);

			wd = (courtSize / 2.0) + border;
			ht = rowHt + border;

			x = outWd + (sideA ? (courtSize + net) : (courtSize - wd)) + (tablet ? 0 : DXDevice.SafeArea().Left);
			y = court.Y + (court.Height / 2.0) - ht;

			// ANDROID: Account for teambar
			if ( wide )
			{
				x += teambarSize;
			}

			layout.SetBounds( this, x, y, wd, (ht * 2) );
		}
		// Portrait
		else
		{
			AddStarColumns( 2 );

			Add( libero1, 0, 0 );
			Add( libero2, 1, 0 );

			double border = tablet ? 4 : 0;
			Padding = tablet ? new Thickness( 0, 0, border, border ) : 2;

			double outHt = ((court.Height - (courtSize * 2) - net) / 2.0);
			double colWd = (courtSize / 3.0);

			wd = colWd + border;
			ht = (courtSize / 2.0) + border;

			x = court.X + (court.Width / 2.0) - wd;
			y = statusHt + outHt + (sideA ? (courtSize + net) : (courtSize - ht));

			y += wide ? teambarSize : 0;

			layout.SetBounds( this, x, y, (wd * 2), ht );
		}

		IsVisible = true;
	}

	// Hides overlay
	public void Hide()
	{
		IsVisible = false;
	}

	/* Layout */

	// Redraws cell
	public override void UpdateLayout( LayoutType type )
	{
		// Update children
		libero1.UpdateLayout( type );
		libero2.UpdateLayout( type );
	}
}

//
