﻿/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using SkiaSharp;
using SkiaSharp.Views.Maui;

using DXLib.UI;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Draws an individual ball marker on the court canvas with optional translucency. 
 */ 
public class Marker
{
	/* Constants */

	// Colors
	private readonly SKColor MarkerColor = DXColors.Dark1.ToSKColor();
	private readonly SKColor OutlineColor = DXColors.Light4.ToSKColor();

	/* Properties */

	// Action text label
	public string Resource { get => text; set => text = DXString.Get( value ); }

	// Player number (optional)
	public string Number { get; set; }

	// Color/fill
	public SKColor? Color { get; set; }
	public byte Alpha { get; set; }

	public bool IsFilled { get; set; }
	public bool IsSmall { get; set; }

	// Location/size
	public SKPoint Location { get => new( x, y ); set { x = value.X; y = value.Y; } }

	/* Fields */
	private string text;

	private float x;
	private float y;
	private float radius;

	/* Methods */
	public Marker()
	{
		// Defaults
		IsFilled = true;
		Alpha = 255;
	}

	// Draws marker onto parent court canvas
	public void Paint( SKCanvas canvas )
	{
		float size = IsSmall ? CourtView.MarkerSizeSmall : CourtView.MarkerSize;

		radius = (size / 2.0f);

		// Solid circle
		if ( IsFilled )
		{
			using SKPaint fill = new();
			
			fill.Style = SKPaintStyle.Fill;
			fill.Color = Color ?? MarkerColor.WithAlpha( Alpha );
			fill.IsAntialias = true;

			canvas.DrawCircle( x, y, radius, fill );
		}

		float strokeWd = (size * 0.10f);

		// White outline
		using ( SKPaint outline = new() )
		{
			outline.Style = SKPaintStyle.Stroke;
			outline.StrokeWidth = strokeWd;
			outline.Color = OutlineColor.WithAlpha( Alpha );
			outline.IsAntialias = true;
			
			canvas.DrawCircle( x, y, radius, outline );
		}

		bool hasNumber = (Number != null);
		
		// Dynamic text size
		float fontSize = size * (hasNumber ? 0.52f : 0.70f);

		// Custom font 
		SKFont font = DXGraphics.GetFont( DXGraphics.Font.RobotoBold );
		font.Size = fontSize;
		
		// Centered text
		using SKPaint style = new();
		
		style.Color = OutlineColor.WithAlpha( Alpha );
		style.IsStroke = false;
		style.IsAntialias = true;
		
		// Player number
		if ( hasNumber )
		{
			int num = int.Parse( Number );
			
			// Account for single vs double digits
			float xOffset = (fontSize * ((num > 9) ? 0.60f : 0.35f));
			float yOffset = (fontSize * 0.35f);
			
			canvas.DrawText( Number, (x - xOffset), (y + yOffset), font, style );
		}
		// Action label
		else
		{
			float xOffset = (fontSize * 0.34f);
			float yOffset = (fontSize * 0.35f);
			
			canvas.DrawText( (text ?? string.Empty), (x - xOffset), (y + yOffset), font, style );
		}
	}
}

//
