﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Container;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Draws one cell in the PlayerGrid. Each cell contains a PlayerArea as well as a ButtonArea.
 */
public class PlayerCell : DXBorder
{
	/* Constants */
	public const double DividerSize = 2;
	public static readonly Color DividerColor = DXColors.Dark3;

	/* Events */
	public Action<ButtonKey> ButtonTapped { get; set; }

	/* Properties */
	public int Zone { get; set; }

	// Current player
	public bool IsEmpty => playerArea.IsEmpty;
	public bool IsLibero => playerArea.IsLibero;
	public bool IsSetter => playerArea.IsSetter;

	/* Fields */
	private readonly DXGridLayout layout;

	// Controls
	private PlayerArea playerArea;
	private ButtonArea buttonArea;

	// Grid proportions
	private static double playerSize;

	/* Methods */
	public PlayerCell()
	{
		Padding = 0;

		// Simulate Material card
		Color = DXColors.Dark4;
		CornerRadius = 5;

		// Underlying layout
		layout = new DXGridLayout
		{
			IsClippedToBounds = true,

			BackgroundColor = DividerColor,
			Padding = 0
		};

		Content = layout;
	}

	// Post construction initialization
	public void Init( Legacy engine )
	{
		base.Init();
		
		// Button Area
		buttonArea = new ButtonArea
		{
			ButtonTapped = OnButtonTapped,
			AreaTapped = OnAreaTapped
		};

		buttonArea.Init( engine );

		// Player Area
		playerArea = new PlayerArea( Zone )
		{
			AreaTapped = OnAreaTapped
		};

		playerArea.Init( engine, buttonArea );
	}

	// Disables player area buttons while rally in-progress
	public void StartRally( bool start )
	{
		playerArea.StartRally( start );
	}

	/* Get/Set */

	// Returns lineup entry populated with info from this cell
	public LineupEntry GetEntry()
	{
		return playerArea.GetEntry();
	}

	// Updates player currently displayed in player area
	public void SetEntry( LineupEntry entry )
	{
		playerArea.SetEntry( entry );
	}

	/* UI Control */

	// Updates state based on current editing mode
	public void UpdateEditMode()
	{
		playerArea.UpdateEditMode();
	}

	// Shows/hides button area for this cell
	public void ShowButtonArea( bool show )
	{
		buttonArea.Show( show );
	}

	// MOBILE: Closes stat button popup
	public void ClosePopup()
	{
		playerArea.ClosePopup();
	}

	/* Enable */

	// Enables an entire category in button area
	public void EnableCategory( string category )
	{
		buttonArea.EnableCategory( category );
	}

	// Enables specified key within given category
	public void Enable( string category, string key )
	{
		buttonArea.Enable( category, key );
	}

	// Enables '+' popup add button on mobile
	public void EnableAdd()
	{
		playerArea.IsAddDisabled = false;
	}

	/* Disable */

	// Disables all buttons in all categories in button area
	public void DisableAll()
	{
		buttonArea.DisableAll();
	}

	// Disables an entire category in button area
	public void DisableCategory( string category )
	{
		buttonArea.DisableCategory( category );
	}

	// Disables individual button in button area
	public void Disable( string category, string key )
	{
		buttonArea.Disable( category, key );
	}

	// Disables '+' popup add button on mobile
	public void DisableAdd()
	{
		playerArea.IsAddDisabled = true;
	}

	/* Tap State */

	// Forces specified button into sticky tap state
	public void ShowTap( ButtonKey key )
	{
		buttonArea.ShowTap( key );
	}

	// Resets sticky state of specified button
	public void ResetTap( ButtonKey key )
	{
		buttonArea.ResetTap( key );
	}

	/* Event Callbacks */

	// User tapped stat button, callback listener
	private void OnButtonTapped( ButtonKey key )
	{
		key.Zone = Zone;

		key.Player = playerArea.Player;
		key.Position = playerArea.Position;
		key.IsAltNumber = playerArea.IsAltNumber;

		ButtonTapped( key );
	}

	// User tapped player or button area to start lineup editing
	private void OnAreaTapped()
	{
		playerArea.ShowLineupMenu();
	}

	/* Layout */

	// Returns dynamic width of player area based on orientation
	public static double PlayerWd()
	{
		double cellWd = PlayerGrid.CellWd();

		return DXDevice.IsLandscape() ? ((cellWd * (playerSize / 100)) - DividerSize) : cellWd;
	}

	// Returns dynamic height of player area based on orientation
	public static double PlayerHt()
	{
		return DXDevice.GetLayoutType() switch
		{
			LayoutType.Landscape or 
			LayoutType.WideLandscape or 
			LayoutType.MobileLandscape or 
			LayoutType.MobilePortrait => PlayerGrid.CellHt(),
			
			LayoutType.Portrait or 
			LayoutType.WidePortrait => (PlayerGrid.CellHt() * (playerSize / 100)) - DividerSize,
			
			_ => 0
		};
	}

	// Redraws player and button areas
	public override void UpdateLayout( LayoutType type )
	{
		layout.ClearAll();

		base.UpdateLayout( type );

		// Update children
		playerArea.UpdateLayout( type );
		buttonArea.UpdateLayout( type );
	}

	// Landscape (4:3)
	protected override void Landscape()
	{
		layout.RowSpacing = 0;
		layout.ColumnSpacing = DividerSize;

		playerSize = 33;

		// 1 row, 2 columns
		layout.AddStarColumn( playerSize );			// Player Area
		layout.AddStarColumn( 100 - playerSize );	// Button Area

		layout.Add( playerArea, 0, 0 );
		layout.Add( buttonArea, 1, 0 );
	}

	// Widescreen Landscape (16:10)
	protected override void WideLandscape()
	{
		Landscape();
	}

	// Portrait (4:3)
	protected override void Portrait()
	{
		layout.RowSpacing = DividerSize;
		layout.ColumnSpacing = 0;

		playerSize = 30;

		// 2 rows, 1 column
		layout.AddStarRow( playerSize );			// Player Area
		layout.AddStarRow( 100 - playerSize );		// Button Area

		layout.Add( playerArea, 0, 0 );
		layout.Add( buttonArea, 0, 1 );
	}

	// Widescreen Portrait (10:16)
	protected override void WidePortrait()
	{
		Portrait();
	}

	// Mobile Landscape
	protected override void MobileLandscape()
	{
		layout.RowSpacing = 0;
		layout.ColumnSpacing = 0;

		// Full bounds
		layout.Add( playerArea, 0, 0 );
	}

	// Mobile Portrait
	protected override void MobilePortrait()
	{
		layout.RowSpacing = 0;
		layout.ColumnSpacing = 0;

		// Full bounds
		layout.Add( playerArea, 0, 0 );
	}
}

//
