﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

namespace iStatVball3;

/*
 * Encapsulates fields necessary for identifying an individual stat button within a ButtonArea.
 */
public class ButtonKey
{
	/* Properties */

	// Court zone (1-6)
	public int Zone { get; set; }

	// Stat category/key
	public string Category { get; set; }
	public string Key { get; set; }

	// Associated lineup entry
	public Player Player { get; set; }
	public string Position { get; set; }
	public bool IsAltNumber { get; set; }

	public string Number => Player?.GetNumber( IsAltNumber );

	// 0-4 rating or att/err?
	public bool IsRating { get; set; }

	/* Methods */
	public ButtonKey()
	{}

	// Creates deep copy from specified source object
	public ButtonKey( ButtonKey src )
	{
		Zone = src.Zone;

		Category = src.Category;
		Key = src.Key;

		Player = src.Player;
		Position = src.Position;
		IsAltNumber = src.IsAltNumber;

		IsRating = src.IsRating;
	}

	// Returns 0-N rating, or null if ATT/ERR
	public int? GetRating()
	{
		// 0-N rating
		if ( IsRating )
		{
			// Map 0-3 to 0-4
			if ( (Category == Legacy.Passing) && Shell.Settings.IsPass03 )
			{
				switch ( Key )
				{
					case "0": return 0;
					case "1": return 2;
					case "2": return 3;
					case "3": return 4;
				}
			}
			// Direct mapping
			else
			{
				return int.Parse( Key );
			}
		}

		// ATT/ERR
		return null;
	}

	// Returns front/backrow for this key
	public int GetCourtRow()
	{
		return (int) LineupEntry.GetRow( Zone );
	}

	// Maps Legacy key to RallyFlow Result
	public string GetResult()
	{
		return Key switch
		{
			// Error
			Stats.ErrorKey or 
			Stats.Rating0 => Stats.ErrorKey,
			
			// 0-Action
			Stats.AttemptKey => Stats.AttemptKey,
			
			_ => Category switch
			{
				Legacy.Serving => Key is Stats.AceKey or Stats.Rating4 ? Stats.AceKey : Stats.AttemptKey,
				Legacy.Hitting => Stats.KillKey,
				Legacy.Free => Stats.KillKey,
				Legacy.Blocking => (Key == Stats.BlockAssistKey) ? Stats.BlockAssistsKey : Stats.BlockKey,
				Legacy.Setting => Stats.AssistKey,
				Legacy.Defense => Stats.DigKey,

				_ => Stats.AttemptKey,
			}
		};
	}

	// Does this key represent an error?
	public bool IsError()
	{
		return Key switch
		{
			// ERR always an error
			Stats.ErrorKey => true,
			
			// Pass-0 error depends on setting
			Stats.Rating0 => (Category != Legacy.Passing) || Shell.Settings.LegacyPass0,
			
			// Not an error
			_ => false
		};
	}

	// Does this key represent an earned point?
	public bool IsPoint()
	{
		return Category switch
		{
			Legacy.Serving  => Key is Stats.AceKey or Stats.Rating4,
			Legacy.Hitting  => (Key == Stats.KillKey),
			Legacy.Free     => (Key == Stats.KillKey),
			Legacy.Blocking => (Key == Stats.BlockKey),

			_ => false
		};
	}
}

//
