﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Gestures;

using DXLib.Data;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Represents a vertical stack of button rows corresponding to one player cell in the player grid. There is one row for
 * each stat category.
 */
public class ButtonArea : DXGridGestures
{
	/* Constants */

	// Category color map
	private static readonly Dictionary<string,Color> Colors = new()
	{
		{ Legacy.Serving,  DXColors.Positive },			// Serve
		{ Legacy.Hitting,  DXColors.Action },			// Hit
		{ Legacy.Passing,  DXColors.Accent4 },			// Pass
		{ Legacy.Blocking, DXColors.Negative },			// Block
		{ Legacy.Setting,  DXColors.Warn },				// Set
		{ Legacy.Defense,  DXColors.Accent5 },			// Defense
	
		{ Legacy.Free, Color.FromArgb( "00BCD4" ) }		// Free (toggles with Hit)
	};

	/* Events */
	public Action<ButtonKey> ButtonTapped { get; set; }
	public Action AreaTapped { get; set; }

	/* Fields */
	private Dictionary<string,ButtonRow> rows;
	private double rowHt;

	// Parent ref
	private Legacy engine;

	/* Methods */
	public ButtonArea()
	{
		BackgroundColor = PlayerCell.DividerColor;

		// REQUIRED for mobile
		IgnoreSafeArea = true;
		
		Horizontal = LayoutOptions.Fill;
		Vertical = LayoutOptions.Fill;
		
		// Register for events
		Tapped += OnTapped;
	}

	// Post construction initialization (same for all layouts)
	public void Init( Legacy legacy )
	{
		engine = legacy;

		// Spacing
		Padding = 0;

		RowSpacing = 1;
		ColumnSpacing = 0;

		int count = Legacy.CategoryCount;

		// Allocate container
		rows = new Dictionary<string,ButtonRow>( count );
		rowHt = (100.0 / count);

		Dictionary<string,string> titles = DXString.GetLookupTable( "legacy" );

		// Create each category
		foreach ( string key in Legacy.CategoryKeys )
		{
			InitCategory( key, titles[ key ] );
		}
	}

	// Creates button row for specified category
	private void InitCategory( string key, string title )
	{
		Settings settings = Shell.Settings;

		bool rating = false;
		string baseKey = $"legacy.{key}";

		switch ( key )
		{
			// Serving either ATT-ACE-ERR or 0-4 rating
			case Legacy.Serving:
			{
				rating = (settings.LegacyServe == Legacy.Serve04);
				baseKey = rating ? "legacy.serve.rate" : "legacy.serve";
				break;
			}
			// Passing uses either 0-3 or 0-4 rating
			case Legacy.Passing:
			{
				rating = true;
				baseKey = settings.IsPass03 ? "legacy.pass03" : "legacy.pass04";
				break;
			}
		}

		List<DXItem> items = DXString.GetLookupList( baseKey );

		// Default configuration
		ButtonConfig config = new()
		{
            Key = key,
            Title = title,
            IsRating = rating,

            Items = items,
            Color = Colors[ key ]
        };

		// Only Hitting eligible for swipe
		bool swipe = (key == Legacy.Hitting);

        // Create row
        ButtonRow row = new()
		{
            SwipeEnabled = swipe,

			Config = config,
			SwipeConfig = swipe ? GetSwipeConfig() : null,

            ButtonTapped = ButtonTapped,
			RowPressed = OnRowPressed
		};

		row.Init();
		rows.Add( key, row );

		// Equal height rows
		AddStarRow( rowHt );
		Add( row, 0, (RowCount - 1) );
	}

	// Returns special configuration for Free button row
	private static ButtonConfig GetSwipeConfig()
	{
        return new ButtonConfig
        {
            Key = Legacy.Free,
            Title = DXString.GetUpper( "legacy.free" ),
            IsRating = false,

            Items = DXString.GetLookupList( "legacy.free" ),
			Color = Colors[ Legacy.Free ]
        };
    }

    // Shows/hides all buttons while maintaining background color
    public void Show( bool show )
	{
		BackgroundColor = show ? PlayerCell.DividerColor : DXColors.Light2;

		// Hide each row
		foreach ( ButtonRow row in rows.Values )
		{
			row.IsVisible = show;
		}
	}

	/* Enable */

	// Enables all buttons in all categories in button area
	public void EnableAll()
	{
		foreach ( string category in rows.Keys )
		{
			EnableCategory( category );
		}
	}

	// Enables an entire category in button area
	public void EnableCategory( string category )
	{
		rows[ category ].EnableAll();
	}

	// Enables individual button in button area
	public void Enable( string category, string key )
	{
		rows[ category ].Enable( key );
	}

	/* Disable */

	// Disables all buttons in all categories in button area
	public void DisableAll()
	{
		foreach ( string category in rows.Keys )
		{
			DisableCategory( category );
		}
	}

	// Disables an entire category in button area
	public void DisableCategory( string category )
	{
		rows[ category ].DisableAll();
	}

	// Disables individual button in button area
	public void Disable( string category, string key )
	{
		rows[ category ].Disable( key );
	}

	/* Tap State */

	// Forces specified button into sticky tap state
	public void ShowTap( ButtonKey key )
	{
		rows[ key.Category ].ShowTap( key );
	}

	// Resets sticky state of specified button
	public void ResetTap( ButtonKey key )
	{
		string category = (key.Category == Legacy.Free) ? Legacy.Hitting : key.Category;

		// Hitting/Free share same row
		rows[ category ].ResetTap( key );
	}

	/* Event Callbacks */

	// User tapped button area to display player select menu
	private void OnTapped( object sender, MR.Gestures.TapEventArgs args )
	{
		if ( engine.IsEditLineup )
		{
			AreaTapped?.Invoke();
		}
	}

	// User long pressed a button row, override Auto Focus
	private void OnRowPressed()
	{
		EnableAll();
	}
}

//
