﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Gestures;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Implementation of the stat recording interface for the Legacy engine. Primary child containers are the scoreboard
 * and player grid.
 */
public class LegacyEngine : RecordEngine
{
	/* Fields */
	private readonly PlayerGrid playerGrid;

	/* Methods */
	public LegacyEngine( DXAbsoluteGestures layout ) : base( layout )
	{
		// Create children
		Scoreboard = new LegacyScore( this );
		playerGrid = new PlayerGrid
		{
			ButtonTapped = OnButtonTapped
		};

		layout.Add( Scoreboard );
		layout.Add( playerGrid );
	}

	// Configures recording UI for new set
	public override async Task Init( Set set )
	{
		await base.Init( set );

		// UI references
		config.PlayerGrid = playerGrid;

		// Init master state machine
		StateMachine = new LegacyState( config );
		await StateMachine.Init();

        // Swipe config
        MR.Gestures.Settings.MinimumDeltaDistance = 2;
    }

    /* Event Callbacks */

    // Forwards all button area key taps to state machine
    private async void OnButtonTapped( ButtonKey key )
	{
		await ((LegacyState)StateMachine).HandleKey( key );
	}

	/* Layout */

	// Redraws entire Legacy recording UI
	public override void UpdateLayout( LayoutType type )
	{
		LayoutType wideType = type;

		// Record UI has special handling for widescreen
		if ( DXDevice.IsTabletWide )
		{
			wideType = (type == LayoutType.Landscape) ? LayoutType.WideLandscape : LayoutType.WidePortrait;
		}

		base.UpdateLayout( wideType );

		// Update children
		playerGrid.UpdateLayout( wideType );
	}

	// Landscape (4:3) [dynamic]
	protected override void Landscape()
	{
		double ht = (DXDevice.GetScreenHt() - DXDevice.SafeArea().VerticalThickness);
		double scoreHt = (ht * 0.200);

		LayoutCommon( scoreHt );
	}

	// Widescreen Landscape (16:10)
	protected override void WideLandscape()
	{
		LayoutCommon( 114 );
	}

	// Portrait (3:4)
	protected override void Portrait()
	{
		double ht = (DXDevice.GetScreenHt() - DXDevice.SafeArea().VerticalThickness);
		double scoreHt = (ht * 0.188);

		LayoutCommon( scoreHt );
	}

	// Widescreen Portrait (10:16)
	protected override void WidePortrait()
	{
		LayoutCommon( 220 );
	}

	// Layout common to multiple non-mobile orientations
	private void LayoutCommon( double scoreHt )
	{
		Thickness safeArea = DXDevice.SafeArea();
		
		double screenWd = DXDevice.GetScreenWd();
		double screenHt = DXDevice.GetScreenHt();

		double scoreY = safeArea.Top;
		
		ScoreboardWd = screenWd;
		ScoreboardHt = scoreHt;

		// Calc grid size
		double gridY = (scoreY + ScoreboardHt);
		double gridHt = (screenHt - safeArea.Bottom - scoreHt - safeArea.Top);

		// Place views
		layout.SetBounds( Scoreboard, 0, scoreY, ScoreboardWd, ScoreboardHt );
		layout.SetBounds( playerGrid, 0, gridY, screenWd, gridHt );
	}

	// Mobile Landscape
	protected override void MobileLandscape()
	{
		bool ios = DXDevice.IsIOS;
		
		double screenWd = DXDevice.GetScreenWd();
		double screenHt = DXDevice.GetScreenHt();
		
		// Dynamic sizing
		ScoreboardWd = screenWd;
		ScoreboardHt = (screenHt * 0.30);

		double scoreHt = ScoreboardHt + (ios ? 0 : (screenHt * 0.05));
		
		double gridX = 0;
		double gridY = scoreHt;

		double gridWd = screenWd;
		double gridHt = (screenHt - scoreHt);
		
		// Place views
		layout.SetBounds( Scoreboard, 0, 0, screenWd, scoreHt );
		layout.SetBounds( playerGrid, gridX, gridY, gridWd, gridHt );
	}

	// Mobile Portrait
	protected override void MobilePortrait()
	{
		bool ios = DXDevice.IsIOS;
		
		double screenWd = DXDevice.GetScreenWd();
		double screenHt = DXDevice.GetScreenHt();

		// Dynamic sizing
		ScoreboardWd = screenWd;
		ScoreboardHt = (screenHt * (ios ? 0.365 : 0.345));

		double gridY = ScoreboardHt;
		double gridHt = (screenHt - ScoreboardHt);

		// Place views
		layout.SetBounds( Scoreboard, 0, 0, screenWd, ScoreboardHt );
		layout.SetBounds( playerGrid, 0, gridY, screenWd, gridHt );
	}
}

//
