﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

namespace iStatVball3;

/*
 * Handles arrangement of the six players currently on court. There are two copies of the arrangement, one in rotation
 * order, the other implementing the Smart Locations feature.
 */
public class LegacyCourt
{
	/* Fields */
	private readonly List<LineupEntry> lineupEntries;

	// SmartLocations
	private readonly LegacySmart smart;

	// External ref
	private readonly PlayerGrid grid;

	/* Methods */
	public LegacyCourt( PlayerGrid grid )
	{
		this.grid = grid;

		// Allocate container
		lineupEntries = new List<LineupEntry>( Lineup.BaseEntries );

		// MUST start with blank entries
		for ( int i = 0; i < Lineup.BaseEntries; i++ )
		{
			lineupEntries.Add( new LineupEntry() );
		}

		// Prep SmartLocations
		smart = new LegacySmart();
	}

	// Called at start of every rally
	public void StartRally( bool start )
	{
		grid.StartRally( start );
	}

	// Updates display with lineup entries in SmartLocations order
	public void UpdateSmart()
	{
		List<LineupEntry> smartEntries = smart.GetSmartEntries( lineupEntries );
		grid.SetEntries( smartEntries );
	}

	/* Get */

	// Returns list of players currently in backrow (zones 5-6-1)
	public IList<LineupEntry> GetBackrow()
	{
		return new List<LineupEntry>( Lineup.RowEntries )
		{
			// 5, 6, 1
			lineupEntries[ 4 ], lineupEntries[ 5 ], lineupEntries[ 0 ]
		};
	}

	// Returns libero currently on court, if any
	public LineupEntry GetLibero()
	{
		return lineupEntries.FirstOrDefault( entry => entry.IsLibero );
	}

	// Returns designated setter on court
	public LineupEntry GetSetter()
	{
		// Zone 1
		LineupEntry entry = GetEntry( 1 );

		if ( entry.IsSetter )
		{
			return entry;
		}

		// Zones 6,5,4,3,2
		for ( int i = (Lineup.BaseEntries - 1); i > 0; i-- )
		{
			entry = lineupEntries[i];

			// Might be blanks
			if ( entry is { IsSetter: true } )
			{
				return entry;
			}
		}

		// No setter found
		return null;
	}

	// Returns lineup entry at specified zone (1-6)
	public LineupEntry GetEntry( int zone )
	{
		int index = (zone - 1);

		return lineupEntries[ index ];
	}

	/* Set */

	// Sets lineup entry for all zones
	public void SetEntries( List<LineupEntry> entries, int rotation, bool update = true )
	{
		bool blank = (entries.Count == 0);

		// Populate each cell
		for ( int i = 0; i < Lineup.BaseEntries; i++ )
		{
			// Adjust for rotation
			int lineupIndex = Lineup.GetLineupIndex( i, rotation );
			int zone = (i + 1);

			// May be blank
			LineupEntry entry = blank ? null : entries[ lineupIndex ];
			SetEntry( zone, entry, update );
		}
	}

	// Sets lineup entry for specified zone
	public void SetEntry( int zone, LineupEntry entry, bool update = true )
	{
		int index = (zone - 1);

		// Deep copy
		lineupEntries[ index ] = new LineupEntry( entry )
		{
			Zone = zone
		};

		// Optionally update grid display
		if ( update )
		{
			grid.SetEntry( zone, entry );
		}
	}
}

//
