﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Card;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Card page for list of Venues within a parent Season.
 */
public class VenuePage : CardPage
{
	/* Methods */
	public VenuePage( Season season ) : base( season )
	{
		Title = DXString.Get( "venue.plural" );
	}

	/* Abstracts */

	// Displays all venue cards
	public override void LoadCards()
	{
		Clear();

		// Required for permissions
		User user = Shell.CurrentUser;

		if ( data is Season season )
		{
			var venues = season.Venues.OrderBy( v => v.Name );

			// Add card for each venue
			foreach ( Venue venue in venues )
			{
				CardSmall card = new()
				{
					Data = venue,

					DefaultIcon = "venue",
					ImageUrl = venue.ImageUrl,
					ImageColor = season.Color,

					Title = venue.Name,
					SubTitle = venue.Location ?? venue.Description,

					HasAnalyze = true,

					IsEditEnabled = Venue.CanEdit( user ),
					IsAnalyzeEnabled = venue.CanAnalyze( user ),

					IsTappable = false,
					AnalyzeTapped = OnAnalyzeTapped,
					EditTapped = OnEditTapped
				};

				card.Init();
				
				// 'Won-Lost-Tied'
				card.UpdateRecord( venue.Won, venue.Lost, venue.Tied );

				AddCard( card );
			}

			// Add card for creating new objects
			if ( Venue.CanCreate( season, user ) )
			{
				AddNewCard( "venue", CardSmall.CardHt );
			}
		}
	}

	/* Event Callbacks */

	// Called back when 'Add New' tapped
	protected override void OnNewCardTapped( DXCard card )
	{
		Season season = data as Season;
		VenueForm form = new( null, season );

		// Open in create mode
		Shell.Instance.ShowForm( form );
	}

	// Called back when card analyze button tapped
	private async void OnAnalyzeTapped( DXCard card )
	{
		if ( (card.Data is Venue venue) && (data is Season season) )
		{
			Shell shell = Shell.Instance;

			// Show dashboard metrics
			shell.DashboardForm.Init( new DashboardConfig
			{
				Scope = DataConfig.VenueScope,
				ScopeName = venue.Name,
				ScopeObject = venue,

				Organization = season.Organization,

				Team1 = season.Team,
				Team2 = null
			});

			await shell.ShowDashboardForm( venue.Name );
		}
	}

	// Called back when card edit button tapped
	private void OnEditTapped( DXCard card )
	{
		Venue venue = card.Data as Venue;
		Season season = data as Season;

		VenueForm form = new( venue, season );

		// Open in edit mode
		Shell.Instance.ShowForm( form );
	}
}

//
