﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Form;
using DXLib.UI.Form.Control;
using DXLib.UI.Form.Message;

namespace iStatVball3;

/*
 * CRUD data entry form for a Venue.
 */
public sealed class VenueForm : DXForm
{
	/* Fields */
	private readonly Season season;

	/* Methods */
	public VenueForm( Venue venue, Season season ) : base( venue, "venue" )
	{
		this.season = season;

		header.Title = HasData ? venue.Name : CreateHeader();
		imagePanel.Color = season.Color;

		/* Required */

		// Name
		AddControl( new DXTextField
		{
			Key = "name",
			Title = "form.name",
			Text = venue?.Name,
			MinLength = 1,
			MaxLength = 64,
			Type = DXTextField.TextType.CapitalizeWords,
			Hint = DXFormControl.HintType.RequiredMin,
			Help = "venue.name"
		}, true, true );

		string courtType = (HasData && (venue?.CourtType != null)) ? venue.CourtType : Venue.DefaultKey;

		bool wood = (courtType == Venue.WoodKey);
		bool tile = (courtType == Venue.TileKey);

		// Court type
		AddControl( new DXSelectorField
		{
			Key = "court.type",
			Title = "venue.court.type",
			Items = "venue.court.type",
			SelectedItem = courtType,
			Help = "venue.court.type"
		}, true, true );

		Color lineColor = HasData ? venue?.LineColor : Venue.DefaultLineColor;

		// Line color
		AddControl( new DXColorPickerField
		{
			IsVisible = (wood || tile),

			Key = "court.line",
			Title = "venue.court.line",
			PickedValue = lineColor?.ToUint(),
			DefaultColor = Venue.DefaultLineColor,
			AllowWhite = true,
			Help = "venue.court.line"
		}, true, true );

		Color innerColor = HasData ? venue?.InnerColor : Venue.DefaultInnerColor;

		// Inner color
		AddControl( new DXColorPickerField
		{
			IsVisible = tile,

			Key = "court.inner",
			Title = "venue.court.inner",
			PickedValue = innerColor?.ToUint(),
			DefaultColor = Venue.DefaultInnerColor,
			AllowWhite = true,
			Help = "venue.court.inner"
		}, true, true );

		Color outerColor = HasData ? venue?.OuterColor : Venue.DefaultOuterColor;

		// Outer color
		AddControl( new DXColorPickerField
		{
			IsVisible = tile,

			Key = "court.outer",
			Title = "venue.court.outer",
			PickedValue = outerColor?.ToUint(),
			DefaultColor = Venue.DefaultOuterColor,
			AllowWhite = true,
			Help = "venue.court.outer"
		}, true, true );

		/* Optional */

		// Description
		AddControl( new DXTextField
		{
			Key = "desc",
			Title = "form.desc",
			Text = venue?.Description,
			MinLength = 1,
			MaxLength = 256,
			Help = "venue.desc"
		}, false, false );

		// Location
		AddControl( new DXTextField
		{
			Key = "location",
			Title = "form.location",
			Text = venue?.Location,
			MinLength = 1,
			MaxLength = 64,
			Type = DXTextField.TextType.CapitalizeWords,
			Help = "venue.location"
		}, false, false );

		// Notes
		AddControl( new DXEditorField
		{
			Key = "notes",
			Title = "form.notes",
			Text = venue?.Notes,
			MaxLength = 1024,
			Help = "venue.notes"
		}, false, false );

		// Image
		SetImage( venue?.ImageUrl );

		// Control initialization
		Init();
	}

	/* Event Callbacks */

	// User confirmed delete
	protected override async void OnDeleteConfirmed()
	{
		base.OnDeleteConfirmed();

		if ( data is Venue venue )
		{
			// Cascading delete
			await venue.Delete( true );

			// Refresh UI
			Shell.Instance.HideForm();
		}
	}

	// Used confirmed cancel
	protected override void OnCancelConfirmed()
	{
		base.OnCancelConfirmed();

		Shell.Instance.HideForm();
	}

	// Used tapped save
	protected override async void OnSaveTapped()
	{
		base.OnSaveTapped();

		// Retrieve fields
		string name = GetString( "name", true );

		string desc = GetString( "desc", false );
		string location = GetString( "location", false );
		string notes = GetString( "notes", false );

		// Custom court
		string type = GetString( "court.type", true );
		long line = GetColor( "court.line", true );
		long inner = GetColor( "court.inner", true );
		long outer = GetColor( "court.outer", true );

		Venue venue;

		// Update existing object
		if ( HasData )
		{
			venue = data as Venue;

			if ( venue != null )
			{
				venue.Name = name;

				venue.Description = desc;
				venue.Location = location;
				venue.Notes = notes;

				// Custom court
				venue.CourtType = type;
				venue.LineColorValue = line;
				venue.InnerColorValue = inner;
				venue.OuterColorValue = outer;

				// Image
				await SaveImage( venue );

				// Persist
				await venue.Update();
			}
		}
		// Create new object
		else
		{
			venue = new Venue
			{
				Name = name,

				Description = desc,
				Location = location,
				Notes = notes,

				// Custom court
				CourtType = type,
				LineColorValue = line,
				InnerColorValue = inner,
				OuterColorValue = outer,

				// Set parent
				SeasonId = season.UniqueId,
				Season = season
			};

			// Image
			await SaveImage( venue );

			// Add to parent
			season.Venues.Add( venue );

			// Persist
			await venue.Create();
		}

		// Refresh UI
		Shell.Instance.HideForm( venue );
	}

	/* Dynamic Fields */

	// Called back when any underlying control value changes
	protected override void OnControlChanged( object sender, DXFormChangedMessage msg )
	{
		base.OnControlChanged( sender, msg );

		DXFormControl control = msg.Value as DXFormControl;
		
		switch ( control?.Key )
		{
			case "court.type": ChangeType( control ); break;
		}
	}

	// User changed court type, update display of color selector(s)
	private void ChangeType( DXFormControl control )
	{
		DXColorPickerField line = GetControl( "court.line", true ) as DXColorPickerField;
		DXColorPickerField inner = GetControl( "court.inner", true ) as DXColorPickerField;
		DXColorPickerField outer = GetControl( "court.outer", true ) as DXColorPickerField;

		line!.IsVisible = false;
		inner!.IsVisible = false;
		outer!.IsVisible = false;

		// Start with default colors
		line.PickedColor = Venue.DefaultLineColor;
		inner.PickedColor = Venue.DefaultInnerColor;
		outer.PickedColor = Venue.DefaultOuterColor;

		string type = control.GetObject() as string;

		switch ( type )
		{
			// Wood customizes line color only
			case Venue.WoodKey:
			{
				line.IsVisible = true;
				break;
			}
			// Tile customizes all colors
			case Venue.TileKey:
			{
				line.IsVisible = true;
				inner.IsVisible = true;
				outer.IsVisible = true;
				break;
			}
		}
	}
}

//
