﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Card;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Card page for list of Tournaments within a parent Season.
 */
public class TournamentPage : CardPage
{
	/* Methods */
	public TournamentPage( Season season ) : base( season )
	{
		Title = DXString.Get( "tournament.plural" );
	}

	/* Abstracts */

	// Displays all tournament cards
	public override void LoadCards()
	{
		Clear();

		// Required for permissions
		User user = Shell.CurrentUser;

		if ( data is Season season )
		{
			// Sort newest first
			List<Tournament> tournaments = season.Tournaments.OrderByDescending( t => t.Created ).ToList();

			// Add card for each tournament
			foreach ( Tournament tournament in tournaments )
			{
				CardSmall card = new( true, false )
				{
					Data = tournament,

					DefaultIcon = "tournament",
					ImageUrl = tournament.ImageUrl,
					ImageColor = season.Color,

					SuperTitle = tournament.Result,
					Title = tournament.Name,
					SubTitle = tournament.GetDateRange(),

					Count = tournament.Matches.Count,
					CountTitle = "match",

					HasAnalyze = true,

					IsEditEnabled = Tournament.CanEdit( user ),
					IsAnalyzeEnabled = tournament.CanAnalyze( user ),

					IsTappable = true,

					CardTapped = OnCardTapped,
					AnalyzeTapped = OnAnalyzeTapped,
					EditTapped = OnEditTapped
				};

				card.Init();
				
				// 'Won-Lost-Tied'
				card.UpdateRecord( tournament.Won, tournament.Lost, tournament.Tied );

				AddCard( card );
			}

			// Add card for creating new objects
			if ( Tournament.CanCreate( season, user ) )
			{
				AddNewCard( "tournament", CardSmall.CardHt );
			}
		}
	}

	/* Event Callbacks */

	// Called back when 'Add New' tapped
	protected override void OnNewCardTapped( DXCard card )
	{
		Season season = data as Season;
		TournamentForm form = new( null, season );

		// Open in create mode
		Shell.Instance.ShowForm( form );
	}

	// Called back when normal card tapped
	private void OnCardTapped( DXCard card )
	{
		Season season = data as Season;

		if ( card.Data is Tournament tournament )
		{
			CardPage page;

			// Tournament spans multiple days
			if ( tournament.IsMultiDay() )
			{
				page = new TournamentDayPage( season, tournament );
			}
			// Normal matches only
			else
			{
				page = new MatchPage( season, tournament );
			}

			// Navigate to matches page, possibly grouped by day
			Shell.Instance.PageStack.Push( page, tournament.Name );
		}
	}

	// Called back when card analyze button tapped
	private async void OnAnalyzeTapped( DXCard card )
	{
		if ( (card.Data is Tournament tournament) && (data is Season season) )
		{
			Shell shell = Shell.Instance;

			// Show dashboard metrics
			shell.DashboardForm.Init( new DashboardConfig
			{
				Scope = DataConfig.TournamentScope,
				ScopeName = tournament.Name,
				ScopeObject = tournament,

				Organization = season.Organization,

				Team1 = tournament.Season.Team,
				Team2 = null
			});

			await shell.ShowDashboardForm( tournament.Name );
		}
	}

	// Called back when card edit button tapped
	private void OnEditTapped( DXCard card )
	{
		Tournament tournament = card.Data as Tournament;
		Season season = data as Season;

		TournamentForm form = new( tournament, season );

		// Open in edit mode
		Shell.Instance.ShowForm( form );
	}
}

//
