﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Form;
using DXLib.UI.Form.Control;

namespace iStatVball3;

/*
 * CRUD data entry form for a Tournament.
 */
public sealed class TournamentForm : DXForm
{
	/* Fields */
	private readonly Season season;

	/* Methods */
	public TournamentForm( Tournament tournament, Season season ) : base( tournament, "tournament" )
	{
		this.season = season;

		header.Title = HasData ? tournament.Name : CreateHeader();
		imagePanel.Color = season.Color;

		/* Required */

		// Name
		AddControl( new DXTextField
		{
			Key = "name",
			Title = "form.name",
			Text = tournament?.Name,
			MinLength = 1,
			MaxLength = 64,
			Type = DXTextField.TextType.CapitalizeWords,
			Hint = DXFormControl.HintType.RequiredMin,
			Help = "tournament.name"
		}, true, true );

		// Description
		AddControl( new DXTextField
		{
			Key = "desc",
			Title = "form.desc",
			Text = tournament?.Description,
			MinLength = 1,
			MaxLength = 256,
			Help = "tournament.desc"
		}, true, true );

		/* Optional */

		// Venue
		AddControl( new DXSelectorField
		{
			Key = "venue",
			Title = "tournament.venue",
			Objects = season.VenueList,
			SelectedObject = season.GetVenue( tournament?.VenueId ),
			Help = "tournament.venue"
		}, false, false );

		// Result
		AddControl( new DXTextField
		{
			Key = "result",
			Title = "tournament.result",
			Text = tournament?.Result,
			MinLength = 1,
			MaxLength = 32,
			Type = DXTextField.TextType.CapitalizeWords,
			Help = "tournament.result"
		}, false, false );

		// Division
		AddControl( new DXTextField
		{
			Key = "division",
			Title = "tournament.division",
			Text = tournament?.Division,
			MinLength = 1,
			MaxLength = 32,
			Type = DXTextField.TextType.CapitalizeWords,
			Help = "tournament.division"
		}, false, false );

		// Seed
		AddControl( new DXKeypadField
		{
			Key = "seed",
			Title = "tournament.seed",
			Number = tournament?.Seed,
			MinValue = 0,
			MaxValue = 999,
			Hint = DXFormControl.HintType.RequiredRange,
			Help = "tournament.seed"
		}, false, false );

		// Notes
		AddControl( new DXEditorField
		{
			Key = "notes",
			Title = "form.notes",
			Text = tournament?.Notes,
			MaxLength = 1024,
			Help = "tournament.notes"
		}, false, false );

		// Image
		SetImage( tournament?.ImageUrl );

		// Control initialization
		Init();
	}

	/* Event Callbacks */

	// User confirmed delete
	protected override async void OnDeleteConfirmed()
	{
		base.OnDeleteConfirmed();

		if ( data is Tournament tournament )
		{
			// Cascading delete
			await tournament.Delete( true );

			// Refresh UI
			Shell.Instance.HideForm();
		}
	}

	// Used confirmed cancel
	protected override void OnCancelConfirmed()
	{
		base.OnCancelConfirmed();

		Shell.Instance.HideForm();
	}

	// Used tapped save
	protected override async void OnSaveTapped()
	{
		base.OnSaveTapped();

		// Retrieve fields
		string name = GetString( "name", true );
		string desc = GetString( "desc", true );

		string result = GetString( "result", false );
		string division = GetString( "division", false );
		int? seed = GetInt( "seed", false );
		string notes = GetString( "notes", false );

		Venue venue = GetObject( "venue", false ) as Venue;

		Tournament tournament;

		// Update existing object
		if ( HasData )
		{
			tournament = data as Tournament;

			if ( tournament != null )
			{
				tournament.Name = name;
				tournament.Description = desc;

				tournament.Result = result;
				tournament.Division = division;
				tournament.Seed = seed;
				tournament.Notes = notes;

				tournament.VenueId = venue?.UniqueId;

				// Image
				await SaveImage( tournament );

				// Persist
				await tournament.Update();
			}
		}
		// Create new object
		else
		{
			tournament = new Tournament
			{
				Name = name,
				Description = desc,

				Result = result,
				Division = division,
				Seed = seed,
				Notes = notes,

				VenueId = venue?.UniqueId,

				// Set parent
				SeasonId = season.UniqueId,
				Season = season
			};

			// Image
			await SaveImage( tournament );

			// Add to parent
			season.Tournaments.Add( tournament );

			// Persist
			await tournament.Create();
		}

		// Refresh UI
		Shell.Instance.HideForm( tournament );
	}
}

//
