﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Card;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Card page for list of Tournament Days within a parent Tournament.
 */
public class TournamentDayPage : CardPage
{
	/* Fields */
	private readonly Tournament tournament;

	/* Methods */
	public TournamentDayPage( Season season, Tournament tournament ) : base( season )
	{
		this.tournament = tournament;

		Title = DXString.Get( "day.plural" );
	}

	/* Abstracts */

	// Displays all tournament day cards
	public override void LoadCards()
	{
		Clear();

		// Required for permissions
		User user = Shell.CurrentUser;

		if ( data is Season season )
		{
			// Group matches by day
			List<TournamentDay> tournamentDays = tournament.GetTournamentDays();

			// Display card for each day
			foreach ( TournamentDay tournamentDay in tournamentDays )
			{
				CardSmall card = new( false, false )
				{
					Data = tournamentDay,

					DefaultIcon = "day",
					ImageColor = season.Color,

					Title = tournamentDay.Name,
					SubTitle = tournamentDay.DateDisplay,

					TitleSize = 24,
					SubTitleSize = 13,

					Count = tournamentDay.Matches.Count,
					CountTitle = "match",

					HasAnalyze = true,
					HasEdit = false,

					IsTappable = true,
					IsAnalyzeEnabled = tournament.CanAnalyze( user ),

					CardTapped = OnCardTapped,
					AnalyzeTapped = OnAnalyzeTapped,
				};

				card.Init();
				
				// 'Won-Lost-Tied'
				card.UpdateRecord( tournamentDay.Won(), tournamentDay.Lost(), tournamentDay.Tied() );

				AddCard( card );
			}

			// Add card for creating new objects
			if ( Match.CanCreate( season, user ) )
			{
				AddNewCard( "match", CardSmall.CardHt );
			}
		}
	}

	/* Event Callbacks */

	// Called back when 'Add New' tapped
	protected override void OnNewCardTapped( DXCard card )
	{
		Season season = data as Season;
		MatchForm form = new( null, tournament, season );

		// Adding new match (maybe grouped by day)
		Shell.Instance.ShowForm( form );
	}

	// Called back when normal card tapped
	private void OnCardTapped( DXCard card )
	{
		Season season = data as Season;

		if ( card.Data is TournamentDay tournamentDay )
		{
			MatchPage page = new( season, tournament, tournamentDay.Number );

			// Navigate to matches page for selected tournament day
			Shell.Instance.PageStack.Push( page, tournamentDay.Name );
		}
	}

	// Called back when card analyze button tapped
	private async void OnAnalyzeTapped( DXCard card )
	{
		if ( (card.Data is TournamentDay day) && (data is Season season) )
		{
			Shell shell = Shell.Instance;

			// Show dashboard metrics
			shell.DashboardForm.Init( new DashboardConfig
			{
				Scope = DataConfig.TournamentDayScope,
				ScopeName = day.FullName,
				ScopeObject = day,

				Organization = season.Organization,

				Team1 = tournament.Season.Team,
				Team2 = null
			});

			await shell.ShowDashboardForm( day.FullName );
		}
	}
}

//
