﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.Utils;

namespace iStatVball3;

/*
 * Represents an individual day of a tournament. Each tournament day contains a list of matches for that day. NOT a
 * database object.
 */
public class TournamentDay
{
	/* Properties */

	// Day number
	public int Number { get; set; }

	// Calendar day
	public DateTimeOffset Date { get; set; }

	// Child matches
	public List<Match> Matches { get; set; }

	// Parent tournament
	public Tournament Tournament { get; set; }

	/* Ignored */

	// 'Day N'
	public string Name => $"{DXString.Get( "day.singular" )} {Number}";

	// 'Fall Classic- Day N'
	public string FullName => $"{Tournament.Name}- {Name}";

	// 'Month Day, Year'
	public string DateDisplay => DXUtils.MonthFromDate( Date );

	/* Methods */

	// Returns total number matches won for this tournament day
	public int Won()
	{
		return Matches.Sum( match => match.Won ? 1 : 0 );
	}

	// Returns total number matches lost for this tournament day
	public int Lost()
	{
		return Matches.Sum( match => match.Lost ? 1 : 0 );
	}

	// Returns total number matches tied for this tournament day
	public int Tied()
	{
		return Matches.Sum( match => match.Tied ? 1 : 0 );
	}

	/* Analysis */

	// Returns list of all matches for this Tournament Day
	public List<Match> GetMatches()
	{
		return Matches;
	}

	// Aggregates all data for analyzing scope of this Tournament Day
	public async Task<DataStats> Aggregate()
	{
		DataStats stats = new();
	
		// Append stats from all matches within Day
		foreach ( Match match in Matches )
		{
			stats.Add( await match.Aggregate() );
		}
	
		return stats;
	}
	
	// Aggregates all raw summary data for scope of this Tournament Day
	public StatSummary AggregateRaw()
	{
		StatSummary summary = new();
	
		// Aggregate summary data for all matches within Day
		foreach ( Match match in Matches )
		{
			summary.Add( match.AggregateRaw() );
		}
	
		return summary;
	}
}

//
