﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Card;
using DXLib.UI.Alert;

using DXLib.Data;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Card page for list of teams within an organization.
 */ 
public class TeamPage : CardPage
{
	/* Methods */
	public TeamPage( Organization organization ) : base( organization )
	{
		Title = DXString.Get( "team.plural" );
	}

	/* Abstracts */

	// Displays all team cards
	public override void LoadCards()
	{
		Clear();

		Organization organization = data as Organization;

		// Required for permissions
		User user = Shell.CurrentUser;

		List<Team> teams = Team.CanView( user, organization ).OrderBy( t => t.Created ).ToList();

		// Add card for each team
		foreach ( Team team in teams )
		{
			CardLarge card = new()
			{
				Data = team,

				DefaultIcon = "team",
				ImageUrl = team.ImageUrl,
				ImageColor = team.Color,

				SuperTitle = team.GenderName.ToUpper(),
				Title = team.Name,
				SubTitle = team.League,

				Count = team.Seasons.Count,
				CountTitle = "season",

				LinkStatus = Link.GetStatus( team.MediaLinks ),

				HasAnalyze = true,

				IsEditEnabled = Team.CanEdit( user ),
				IsAnalyzeEnabled = team.CanAnalyze( user ),

				CardTapped = OnCardTapped,
				AnalyzeTapped = OnAnalyzeTapped,
				EditTapped = OnEditTapped
			};
			
			card.Init();
			
			AddCard( card );
		}

		// Add card for creating new objects
		if ( Team.CanCreate( organization, user ) )
		{
			AddNewCard( "team", CardLarge.CardHt );
		}
	}

	/* Event Callbacks */

	// Called back when 'Add New' tapped
	protected override void OnNewCardTapped( DXCard card )
	{
		Organization org = data as Organization;
		TeamForm form = new( null, org );

		// Open in create mode
		Shell.Instance.ShowForm( form );
	}

	// Called back when normal card tapped
	private static void OnCardTapped( DXCard card )
	{
		Team team = card.Data as Team;
		SeasonPage page = new( team );
		
		// Navigate to seasons page for selected team
		Shell.Instance.PageStack.Push( page, team?.Name );
	}

	// Called back when card analyze button tapped
	private void OnAnalyzeTapped( DXCard card )
	{
		// Network connection required for career stats
		if ( !DXData.HasConnection() )
		{
			DXAlert.ShowNetworkError( "net.err.career", card.Reset );
		}
		// User already warned about performance
		else if ( DXPreferences.Get( "team.warned", false ) )
		{
			OnAnalyzeConfirmed( card );
		}
		// Warn user
		else
		{
			DXAlert.ShowOkCancel( "analyze.perf", "analyze.perf.msg", () => OnAnalyzeConfirmed( card ), card.Reset );

			// Only warn once
			DXPreferences.Set( "team.warned", true );
		}
	}

	// Called back when card analyze button tapped
	private static async void OnAnalyzeConfirmed( DXCard card )
	{
		if ( card.Data is Team team )
		{
			Shell shell = Shell.Instance;

			// Show dashboard metrics
			shell.DashboardForm.Init( new DashboardConfig
			{
				Scope = "team",
				ScopeName = team.Name,
				ScopeObject = team,

				Organization = team.Organization,

				Team1 = team,
				Team2 = null,
			});

			await shell.ShowDashboardForm( team.Name );
		}
	}

	// Called back when card edit button tapped
	private void OnEditTapped( DXCard card )
	{
		Team team = card.Data as Team;
		Organization org = data as Organization;

		TeamForm form = new( team, org );

		// Open in edit mode
		Shell.Instance.ShowForm( form );
	}
}

//
