﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Form;
using DXLib.UI.Form.Control;

namespace iStatVball3;

/*
 * CRUD data entry form for a Team.
 */
public sealed class TeamForm : LinkForm
{
	/* Fields */
	private readonly Organization organization;

	/* Methods */
	public TeamForm( Team team, Organization organization ) : base( team, "team" )
	{
		this.organization = organization;

		header.Title = HasData ? team.Name : CreateHeader();
		imagePanel.Color = organization.Color;

		/* Required */

		// Gender
		AddControl( new DXSelectorField
		{
			Key = "gender",
			Title = "team.gender",
			Items = "team.gender",
			SelectedItem = team?.Gender,
			Hint = DXFormControl.HintType.Required,
			Help = "team.gender"
		}, true, true );

		// Level (club and high school only)
		if ( organization.IsClub || organization.IsHighSchool )
		{
			AddControl( new DXSelectorField
			{
				Key = "level",
				Title = "team.level",
				Items = organization.IsClub ? "team.level.club" : "team.level.hs",
				SelectedItem = team?.Level,
				Hint = DXFormControl.HintType.Required,
				Help = organization.IsClub ? "team.level.club" : "team.level.hs"
			}, true, true );
		}

		// Name
		AddControl( new DXTextField
		{
			Key = "name",
			Title = "form.name",
			Text = team?.Name,
			MinLength = 2,
			MaxLength = 64,
			Type = DXTextField.TextType.CapitalizeWords,
			Hint = DXFormControl.HintType.RequiredMin,
			Help = "team.name"
		}, true, true );

		// Default to org abbreviation
		string abbrev = HasData ? team?.AbbrevName : organization.Abbreviation;

		// Abbreviation
		AddControl( new DXTextField
		{
			Key = "abbreviation",
			Title = "form.abbrev",
			Text = abbrev,
			MinLength = 1,
			MaxLength = 5,
			Type = DXTextField.TextType.AllCaps,
			Hint = DXFormControl.HintType.RequiredRange,
			Help = "team.abbreviation"
		}, true, true );

		/* Optional */

		// League
		AddControl( new DXTextField
		{
			Key = "league",
			Title = "team.league",
			Text = team?.League,
			MaxLength = 32,
			Type = DXTextField.TextType.CapitalizeWords,
			Help = "team.league"
		}, false, false );

		// Division
		AddControl( new DXTextField
		{
			Key = "division",
			Title = "team.division",
			Text = team?.Division,
			MaxLength = 32,
			Type = DXTextField.TextType.CapitalizeWords,
			Help = "team.division"
		}, false, false );

		// Notes
		AddControl( new DXEditorField
		{
			Key = "notes",
			Title = "form.notes",
			Text = team?.Notes,
			MaxLength = 1024,
			Help = "team.notes"
		}, false, false );

		/* Links */

		// Media Link(s)
		if ( HasData )
		{
			AddControl( new LinkField
			{
				Key = "link.media",
				Title = "team.link.media",
				Links = team?.MediaLinks,
				Help = "team.link.media"
			});
		}

		// Image
		SetImage( team?.ImageUrl );

		// Control initialization
		Init();
	}

	/* Event Callbacks */

	// User confirmed delete
	protected override async void OnDeleteConfirmed()
	{
		base.OnDeleteConfirmed();

		if ( data is Team team )
		{
			// Cascading delete
			await team.Delete( true );

			// Refresh UI
			Shell.Instance.HideForm();
		}
	}

	// Used confirmed cancel
	protected override void OnCancelConfirmed()
	{
		base.OnCancelConfirmed();

		Shell.Instance.HideForm();
	}

	// Used tapped save
	protected override async void OnSaveTapped()
	{
		// Connection required
		if ( !ValidateConnection() )
		{
			return;
		}

		// Retrieve fields
		string gender = GetString( "gender", true );
		string level = GetString( "level", true );
		string name = GetString( "name", true );
		string abbreviation = GetString( "abbreviation", true );

		string league = GetString( "league", false );
		string division = GetString( "division", false );
		string notes = GetString( "notes", false );

		// Link
		List<string> links = GetLinks( "link.media" );

		// Validate unique name
		if ( !HasData || (name != (data as Team)?.Name) )
		{
			if ( organization.GetTeamByName( name ) != null )
			{
				ShowUniqueError( "team", "organization" );
				return;
			}
		}

		base.OnSaveTapped();

		// Update existing object
		if ( HasData )
		{
			if ( data is Team team )
			{
				team.Gender = gender;
				team.Level = level;
				team.Name = name;
				team.Abbreviation = abbreviation;

				team.League = league;
				team.Division = division;
				team.Notes = notes;

				// Image
				await SaveImage( team );

				// Update Media link(s)
				await UpdateLinks( team, links, team.MediaLinks, CreateKeys( Permission.LevelType.Media, team ) );

				// Persist
				await team.Update();
			}
		}
		// Create new object
		else
		{
			Team team = new()
			{
				Gender = gender,
				Level = level,
				Name = name,
				Abbreviation = abbreviation,

				League = league,
				Division = division,
				Notes = notes,

				// Set parent
				OrganizationId = organization.UniqueId,
				Organization = organization
			};

			// Image
			await SaveImage( team );

			// Grant owner permission
			await Permission.CreateOwner( Shell.CurrentUser, team );

			// Add Media link(s)
			await CreateLinks( team, links, CreateKeys( Permission.LevelType.Media, team ) );

			// Add to parent
			organization.Teams.Add( team );

			// Persist
			await team.Create();
		}

		// Refresh UI
		Shell.Instance.HideForm();
	}
}

//
