﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Card;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Card page for list of Statisticians within a parent Season.
 */
public class StatisticianPage : CardPage
{
	/* Methods */
	public StatisticianPage( Season season ) : base( season )
	{
		Title = DXString.Get( "statistician.plural" );
	}

	/* Abstracts */

	// Displays all statistician cards
	public override void LoadCards()
	{
		Clear();

		// Required for permissions
		User user = Shell.CurrentUser;

		if ( data is Season season )
		{
			var statisticians = season.Statisticians.OrderBy( s => s.LastName );

			// Add card for each statistician
			foreach ( Statistician statistician in statisticians )
			{
				CardSmall card = new()
				{
					Data = statistician,

					DefaultIcon = "statistician",
					ImageUrl = statistician.ImageUrl,
					ImageColor = season.Color,

					Title = statistician.FullName,
					SubTitle = statistician.TypeName,

					LinkStatus = statistician.Link?.Status,
					HasAnalyze = true,

					IsEditEnabled = Statistician.CanEdit( user ),
					IsAnalyzeEnabled = statistician.CanAnalyze( user ),

					IsTappable = false,
					AnalyzeTapped = OnAnalyzeTapped,
					EditTapped = OnEditTapped
				};

				card.Init();
				
				// 'Won-Lost-Tied'
				card.UpdateRecord( statistician.Won, statistician.Lost, statistician.Tied );

				AddCard( card );
			}

			// Add card for creating new objects
			if ( Statistician.CanCreate( season, user ) )
			{
				AddNewCard( "statistician", CardSmall.CardHt );
			}
		}
	}

	/* Event Callbacks */

	// Called back when 'Add New' tapped
	protected override void OnNewCardTapped( DXCard card )
	{
		Season season = data as Season;
		StatisticianForm form = new( null, season );

		// Open in create mode
		Shell.Instance.ShowForm( form );
	}

	// Called back when card analyze button tapped
	private async void OnAnalyzeTapped( DXCard card )
	{
		if ( (card.Data is Statistician statistician) && (data is Season season) )
		{
			Shell shell = Shell.Instance;

			// Show dashboard metrics
			shell.DashboardForm.Init( new DashboardConfig
			{
				Scope = DataConfig.StatisticianScope,
				ScopeName = statistician.FullName,
				ScopeObject = statistician,

				Organization = season.Organization,

				Team1 = season.Team,
				Team2 = null
			});

			await shell.ShowDashboardForm( statistician.FullName );
		}
	}

	// Called back when card edit button tapped
	private void OnEditTapped( DXCard card )
	{
		Statistician statistician = card.Data as Statistician;
		Season season = data as Season;

		StatisticianForm form = new( statistician, season );

		// Open in edit mode
		Shell.Instance.ShowForm( form );
	}
}

//
