﻿/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using DXLib.UI.Form;
using DXLib.UI.Form.Control;

using DXLib.Data;

namespace iStatVball3;

/*
 * CRUD data entry form for a Statistician.
 */
public sealed class StatisticianForm : LinkForm
{
	/* Fields */
	private readonly Season season;

	/* Methods */
	public StatisticianForm( Statistician statistician, Season season ) : base( statistician, "statistician" )
	{
		this.season = season;

		header.Title = HasData ? statistician.FullName : CreateHeader();
		imagePanel.Color = season.Color;

		/* Required */

		// Type
		AddControl( new DXSelectorField
		{
			Key = "type",
			Title = "form.type",
			Items = "statistician.type",
			SelectedItem = statistician?.Type,
			Hint = DXFormControl.HintType.Required,
			Help = "statistician.type"
		}, true, true );

		// First Name
		AddControl( new DXTextField
		{
			Key = "first",
			Title = "statistician.first",
			Text = statistician?.FirstName,
			MinLength = 1,
			MaxLength = 32,
			Type = DXTextField.TextType.CapitalizeWords,
			Hint = DXFormControl.HintType.RequiredMin,
			Help = "statistician.first"
		}, true, true );

		// Last Name
		AddControl( new DXTextField
		{
			Key = "last",
			Title = "statistician.last",
			Text = statistician?.LastName,
			MinLength = 1,
			MaxLength = 32,
			Type = DXTextField.TextType.CapitalizeWords,
			Hint = DXFormControl.HintType.RequiredMin,
			Help = "statistician.last"
		}, true, true );

		/* Optional */

		// Nickname
		AddControl( new DXTextField
		{
			Key = "nickname",
			Title = "statistician.nickname",
			Text = statistician?.Nickname,
			MinLength = 1,
			MaxLength = 64,
			Type = DXTextField.TextType.CapitalizeWords,
			Help = "statistician.nickname"
		}, false, false );

		// Notes
		AddControl( new DXEditorField
		{
			Key = "notes",
			Title = "form.notes",
			Text = statistician?.Notes,
			MaxLength = 1024,
			Help = "statistician.notes"
		}, false, false );

		/* Links */

		// Statistician link
		if ( HasData )
		{
			AddControl( new LinkField
			{
				IsDisabled = !HasData || !DXData.HasConnection(),

				Key = "link",
				Title = "statistician.link",
				Link = statistician?.Link,
				Help = "statistician.link"
			});
		}

		// Image
		SetImage( statistician?.ImageUrl );

		// Control initialization
		Init();
	}

	/* Event Callbacks */

	// User confirmed delete
	protected override async void OnDeleteConfirmed()
	{
		base.OnDeleteConfirmed();

		if ( data is Statistician statistician )
		{
			// Cascading delete
			await statistician.Delete( true );

			// Refresh UI
			Shell.Instance.HideForm();
		}
	}

	// Used confirmed cancel
	protected override void OnCancelConfirmed()
	{
		base.OnCancelConfirmed();

		Shell.Instance.HideForm();
	}

	// Used tapped save
	protected override async void OnSaveTapped()
	{
		base.OnSaveTapped();

		// Retrieve fields
		string type = GetString( "type", true );
		string firstName = GetString( "first", true );
		string lastName = GetString( "last", true );

		string nickname = GetString( "nickname", false );
		string notes = GetString( "notes", false );

		// Link
		string link = GetLink( "link" );

		Statistician statistician;

		// Update existing object
		if ( HasData )
		{
			statistician = data as Statistician;

			if ( statistician != null )
			{
				statistician.Type = type;
				statistician.FirstName = firstName;
				statistician.LastName = lastName;

				statistician.Nickname = nickname;
				statistician.Notes = notes;

				// Image
				await SaveImage( statistician );

				// Update account link
				statistician.Link = await UpdateLink( link, statistician.Link,
				CreateKeys( Permission.LevelType.Statistician, season.Team ) );

				// Persist
				await statistician.Update();
			}
		}
		// Create new object
		else
		{
			statistician = new Statistician
			{
				Type = type,
				FirstName = firstName,
				LastName = lastName,

				Nickname = nickname,
				Notes = notes,

				// Set parent
				SeasonId = season.UniqueId,
				Season = season
			};

			// Image
			await SaveImage( statistician );

			// Add account link
			statistician.Link = await CreateLink( link, CreateKeys( Permission.LevelType.Statistician, season.Team ) );

			// Add to parent
			season.Statisticians.Add( statistician );

			// Persist
			await statistician.Create();
		}

		// Refresh UI
		Shell.Instance.HideForm( statistician );
	}
}

//
