﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using System.Text.Json.Serialization;
using Plugin.Firebase.Firestore;

namespace iStatVball3;

/*
 * Used to restore state for undo/restore operations while a set is in-progress. All state objects can be deleted once
 * the set is ended. Persisted as a map within the parent Stat NOT as a root level collection.
 */ 
public class StatState : IFirestoreObject
{
	/* Properties */

	[FirestoreProperty("State")] public string State { get; set; }
	[FirestoreProperty("StateSelector")] public string StateSelector { get; set; }

	// Court sides
	[FirestoreProperty("BallSide")] public int BallSide { get; set; }
	[FirestoreProperty("ServeSide")] public int ServeSide { get; set; }

	// Score, rotation, etc based on court side A/B
	[FirestoreProperty("ScoreA")] public int ScoreA { get; set; }
	[FirestoreProperty("ScoreB")] public int ScoreB { get; set; }

	[FirestoreProperty("RotationA")] public int RotationA { get; set; }
	[FirestoreProperty("RotationB")] public int RotationB { get; set; }

	[FirestoreProperty("TimeoutsA")] public int TimeoutsA { get; set; }
	[FirestoreProperty("TimeoutsB")] public int TimeoutsB { get; set; }

	[FirestoreProperty("SubsA")] public int SubsA { get; set; }
	[FirestoreProperty("SubsB")] public int SubsB { get; set; }

	[FirestoreProperty("LiberoA")] public int LiberoA { get; set; }
	[FirestoreProperty("LiberoB")] public int LiberoB { get; set; }

	// Point state
	[FirestoreProperty("Touches")] public int Touches { get; set; }
	[FirestoreProperty("Transitions")] public int Transitions { get; set; }

	[FirestoreProperty("ServeCount")] public int ServeCount { get; set; }

	// Used to restore fault flyout
	[FirestoreProperty("FaultKey")] public string FaultKey { get; set; }

	// Entries (array of maps)
	[FirestoreProperty("Entries1")] public IList<LineupEntry> Entries1 { get; set; }

	// References
	[FirestoreProperty("TeamAId")] public string TeamAId { get; set; }
	[FirestoreProperty("TeamBId")] public string TeamBId { get; set; }

	/* Ignored */
	[JsonIgnore] public bool HasEntries => Entries1.Count > 0;

	/* Methods */
	public StatState()
	{
		// Allocate containers
		Entries1 = new List<LineupEntry>();
	}

	// Populates all lineup entry FKs
	public void Populate( Season season )
	{
		foreach ( LineupEntry entry in Entries1 )
		{
			entry?.Populate( season );
		}
	}

	// Saves list of all entries from specified lineup(s)
	public void SaveLineups( List<LineupEntry> entries1 )
	{
		Lineup.CopyEntries( entries1, Entries1 );
	}

	// Changes specified previous team to new team (no db)
	public void UpdateTeam( string oldId, string newId )
	{
		// Could be either court side (null always anon opponent)
		if ( (TeamAId == oldId) || (TeamAId == null) )
		{
			TeamAId = newId;
		}
		else if ( (TeamBId == oldId) || (TeamBId == null) )
		{
			TeamBId = newId;
		}
	}
}

//
